(ns doccla.oth-client.chat.thread-messages
  (:require [clj-http.client :as client]
            [doccla.oth-client.schemas :as schemas]
            [doccla.oth-client.utils :as utils]
            [malli.clj-kondo :as clj-kondo]
            [malli.core :as m]
            [malli.instrument :as mi]))

(def query-params-schema
  [:map
   [:offset {:optional true} int?]
   [:max {:optional true} int?]])

(def attachments-links-schema
  [:map
   [:full {:optional true} [:re schemas/url-regex]]
   [:thumbnail {:optional true} [:re schemas/url-regex]]])

(def message-sender-schema
  [:map
   [:name string?]
   [:type [:enum "organization" "patient"]]])

(def message-schema
  [:map
   [:body {:optional true} string?]
   [:links [:map
            [:attachments {:optional true} [:vector [:maybe attachments-links-schema]]]
            [:organization [:re schemas/url-regex]]
            [:patient [:re schemas/url-regex]]
            [:message [:re schemas/url-regex]]
            [:thread [:re schemas/url-regex]]]]
   [:read-at {:optional true} [:re schemas/iso-8601-regex]]
   [:sender message-sender-schema]
   [:timestamp {:optional true} [:re schemas/iso-8601-regex]]])

(def thread-messages-schema
  [:map
   [:total int?]
   [:max int?]
   [:offset int?]
   [:results [:vector [:maybe message-schema]]]
   [:links [:map
            [:self [:re schemas/url-regex]]
            [:next {:optional true} [:re schemas/url-regex]]
            [:previous {:optional true} [:re schemas/url-regex]]]]])

(m/=> get-thread-messages [:=>
                           [:cat schemas/opts-schema [:or string? int?] query-params-schema]
                           [:or schemas/error-schema (schemas/success-schema thread-messages-schema)]])

(defn ^:mockable get-thread-messages
  "Retrieve a list of chat messages for a given thread uuid"
  [opts thread-id params]
  (let [res (client/get (str (:base-url opts) "/chat/threads/" thread-id "/messages")
                        (assoc (utils/opts->request opts)
                               :query-params params))
        post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                    (f thread-messages-schema data utils/prune-map-transformer)))]
    (utils/->output [200] post-processor res)))

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope #{:input}
                 :report utils/input-validate-fail!})
(clj-kondo/emit!)
;; Enable mocks
(utils/make-mockable)

(comment
 ;; Example usage
  #_:clj-kondo/ignore
  (get-thread-messages {:base-url "https://doccla-dev.oth.io"
                        :validate-output? true
                        :auth {:type :id-secret :id "" :secret ""}}
                       "a63e808c-b530-4c8f-b903-788146b2c03e"
                       {:max 300}))