(ns doccla.oth-client.notifications.subscriptions
  (:require
   [clj-http.client :as client]
   [doccla.oth-client.utils :as utils]
   [doccla.oth-client.schemas :as schemas]
   [malli.clj-kondo :as clj-kondo]
   [malli.core :as m]
   [malli.instrument :as mi]))

(def delivery-schema
  [:or
   [:map
    [:type [:enum "webhook"]]
    [:url string?]
    [:secret string?]]
   [:map
    [:type [:enum "push"]]
    [:patient-url string?]
    [:device-token string?]
    [:device-os [:enum "iOS" "Android"]]]])

(def get-subscriptions-input-schema
  [:map
   [:offset {:optional true} int?]
   [:max {:optional true} int?]])

(def get-subscriptions-response-schema
  [:map
   [:results
    [:vector
     [:map
      [:delivery delivery-schema]
      [:event-types
       [:vector string?]]
      [:links
       [:map
        [:subscription string?]]]]]]
   [:links
    [:map
     [:self string?]
     [:previous {:optional true} string?]
     [:next {:optional true} string?]]]
   [:max int?]
   [:offset int?]
   [:total int?]])

(def create-or-update-body-schema
  [:map
   [:delivery delivery-schema]
   [:event-types
    [:vector string?]]])

(def create-or-update-response-schema
  [:map
   [:delivery delivery-schema]
   [:event-types
    [:vector string?]]
   [:links
    [:map
     [:self string?]]]])

(m/=> get-subscriptions [:=>
                         [:cat schemas/opts-schema get-subscriptions-input-schema]
                         [:or schemas/error-schema (schemas/success-schema get-subscriptions-response-schema)]])

(m/=> create-subscription [:=>
                           [:cat schemas/opts-schema create-or-update-body-schema]
                           [:or schemas/error-schema (schemas/success-schema create-or-update-response-schema)]])

(m/=> update-subscription [:=>
                           [:cat schemas/opts-schema create-or-update-body-schema string?]
                           [:or schemas/error-schema (schemas/success-schema create-or-update-response-schema)]])

(m/=> delete-subscription [:=>
                           [:cat schemas/opts-schema string?]
                           [:or schemas/error-schema (schemas/success-schema :nil)]])

(defn post-processor
  [opts schema]
  (fn [data]
    (let [f (if (:validate-output? opts) m/coerce m/decode)]
      (f schema data utils/prune-map-transformer))))

(defn ^:mockable create-subscription
  "Create a new webhook subscription."
  [opts body]
  (let [url (str (:base-url opts) "/notifications/subscriptions")
        req (utils/opts->request-with-body opts body)
        res (client/post url req)]
    (utils/->output [201] (post-processor opts create-or-update-response-schema) res)))

(defn ^:mockable update-subscription
  "Update an existing webhook subscription."
  [opts body subscription-link]
  (let [req (utils/opts->request-with-body opts body)
        res (client/put subscription-link req)]
    (utils/->output [200] (post-processor opts create-or-update-response-schema) res)))

(defn ^:mockable delete-subscription
  "Delete a subscription"
  [opts subscription-link]
  (let [req (utils/opts->request opts)
        res (client/delete subscription-link req)]
    (utils/->output [204] (post-processor opts :nil) res)))

(defn ^:mockable get-subscriptions
  "Get a list of current subscriptions"
  ([opts]
   (get-subscriptions opts {}))
  ([opts params]
   (let [url (str (:base-url opts) "/notifications/subscriptions")
         req (assoc (utils/opts->request opts)
                    :multi-param-style :array
                    :query-params params)
         res (client/get url req)]
     (utils/->output [200] (post-processor opts get-subscriptions-response-schema) res))))

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope #{:input}
                 :report utils/input-validate-fail!})
(clj-kondo/emit!)
;; Enable mocks
(utils/make-mockable)

(comment

  ;; Example usage

  ;; Create subscription
  (let [creds {:id "" :secret ""}
        opts {:base-url "https://doccla-dev.oth.io"
              :validate-output? false
              :auth {:type :id-secret
                     :id (:id creds)
                     :secret (:secret creds)}}
        body {:delivery {:type "webhook"
                         :url "http://example.com/listen"
                         :secret "ever_so_secr3t"}
              :event-types ["measurement.created"
                            "questionnaireResult.created"]}]
    (create-subscription opts body))

  ;; Update subscription
  (let [creds {:id "" :secret ""}
        opts {:base-url "https://doccla-dev.oth.io"
              :validate-output? false
              :auth {:type :id-secret
                     :id (:id creds)
                     :secret (:secret creds)}}
        body {:delivery {:type "webhook"
                         :url "http://example.com/listen"
                         :secret "ever_so_secr3t"}
              :event-types ["measurement.created"
                            "questionnaireResult.created"
                            "questionnaireResult.acknowledged"]}
        subscription-link "https://doccla-dev.oth.io/notifications/subscriptions/7ef06118-3a53-4548-924e-5d54b1332f3a"]
    (update-subscription opts body subscription-link))

  ;; Delete subscription
  (let [creds {:id "" :secret ""}
        opts {:base-url "https://doccla-dev.oth.io"
              :validate-output? false
              :auth {:type :id-secret
                     :id (:id creds)
                     :secret (:secret creds)}}
        subscription-link "https://doccla-dev.oth.io/notifications/subscriptions/7ef06118-3a53-4548-924e-5d54b1332f3a"]
    (delete-subscription opts subscription-link))

  ;; List subscriptions
  (let [creds {:id "" :secret ""}
        opts {:base-url "https://doccla-dev.oth.io"
              :validate-output? false
              :auth {:type :id-secret
                     :id (:id creds)
                     :secret (:secret creds)}}
        params {:max 5
                :offset 0}]
    (get-subscriptions opts params)))