(ns doccla.oth-client.idp2.users.users
  (:require
   [clj-http.client :as client]
   [doccla.oth-client.utils :as utils]
   [doccla.oth-client.schemas :as schemas]
   [malli.clj-kondo :as clj-kondo]
   [malli.core :as m]
   [malli.instrument :as mi]
   [malli.util :as mu]))

(def get-users-params-schema
  [:map
   [:user-type {:optional true} [:enum "patient" "clinician" "guardian"]]
   [:url {:optional true} string?]
   [:offset {:optional true} int?]
   [:max {:optional true} int?]])

(def user-schema-base
  [:map
   [:username string?]
   [:first-name string?]
   [:last-name string?]
   [:email {:optional true} string?]
   [:audit-id string?]])

(def get-users-response-schema
  [:map
   [:users [:vector (mu/merge user-schema-base [:map [:links [:map [:user [:re schemas/url-regex]]]]])]]
   [:offset int?]
   [:max int?]
   [:total int?]
   [:links [:map
            [:self [:re schemas/url-regex]]
            [:previous {:optional true} [:re schemas/url-regex]]
            [:next {:optional true} [:re schemas/url-regex]]]]])

(defn post-processor
  [opts schema]
  (fn [data]
    (let [f (if (:validate-output? opts) m/coerce m/decode)]
      (f schema data utils/prune-map-transformer))))

(m/=> get-users [:=>
                 [:cat schemas/opts-schema get-users-params-schema]
                 [:or schemas/error-schema (schemas/success-schema get-users-params-schema)]])

(defn ^:mockable get-users
  "Retrieve a list of users according to the given parameters."
  [opts params]
  (let [res (client/get (str (:base-url opts) "/idp2/users")
                        (assoc (utils/opts->request opts)
                               :query-params params))]
    (utils/->output [200] (post-processor opts get-users-response-schema) res)))

(def get-user-response-schema
  (mu/merge
   user-schema-base
   [:map
    [:phone {:optional true} string?]
    [:mobile-phone {:optional true} string?]
    [:temporary-password {:optional true} string?]
    [:password-expired boolean?]
    [:account-locked boolean?]
    [:audit-id string?]
    [:status [:enum "active" "inactive"]]
    [:roles
     [:vector
      [:map
       [:name string?]
       [:links
        [:map
         [:role [:re schemas/url-regex]]]]]]]
    [:links [:map [:self [:re schemas/url-regex]] [:password [:re schemas/url-regex]] [:locked [:re schemas/url-regex]]]]]))

(m/=> get-user [:=>
                [:cat schemas/opts-schema [:or :string :int]]
                [:or schemas/error-schema (schemas/success-schema get-user-response-schema)]])

(defn ^:mockable get-user
  "Retrieve a specific user"
  [opts id]
  (let [res (client/get (str (:base-url opts) "/idp2/users/" id)
                        (utils/opts->request opts))]
    (utils/->output [200] (post-processor opts get-user-response-schema) res)))

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope #{:input}
                 :report utils/input-validate-fail!})
(clj-kondo/emit!)
;; Enable mocks
(utils/make-mockable)

(comment
  (require '[clojure.edn :as edn])
  (def creds (edn/read-string (slurp "creds/oth.edn")))
  (def opts {:base-url    "https://doccla-dev.oth.io"
             :validate-output? true
             :auth {:type :id-secret :id (:id creds) :secret (:secret creds)}})

  (get-users opts {:userType "clinician" :url "https://doccla-dev.oth.io/clinician/api/clinicians/127"})
  (get-user opts 4188939))