(ns doccla.oth-client.audit.events
  (:require
   [clj-http.client :as client]
   [doccla.oth-client.utils :as utils]
   [doccla.oth-client.schemas :as schemas]
   [malli.clj-kondo :as clj-kondo]
   [malli.core :as m]
   [malli.instrument :as mi]))

(def change-schema
  [:map
   [:property-name string?]
   [:property-type [:enum "string" "number" "boolean" "array"]]
   [:added {:optional true} [:or number? string? boolean? [:vector any?]]]
   [:removed {:optional true} [:or number? string? boolean? [:vector any?]]]])

(def resource-schema
  [:map
   [:description string?]
   [:type string?]
   [:url [:re schemas/url-regex]]])

(def responsible-schema
  [:map
   [:name string?]
   [:url [:re schemas/url-regex]]])

(def event-schema
  [:map
   [:event-name string?]
   [:event-time [:re schemas/iso-8601-regex]]
   [:event-context {:optional true} string?]
   [:resource resource-schema]
   [:responsible responsible-schema]
   [:where [:re schemas/ip-address-regex]]
   [:changes [:vector change-schema]]
   [:links [:map
            [:resource-log string?]
            [:self {:optional true} [:re schemas/url-regex]]
            [:event {:optional true} [:re schemas/url-regex]]]]])

(def events-schema
  [:map
   [:results [:vector event-schema]]
   [:offset int?]
   [:max int?]
   [:total int?]
   [:links [:map
            [:self [:re schemas/url-regex]]
            [:previous {:optional true} [:re schemas/url-regex]]
            [:next {:optional true} [:re schemas/url-regex]]]]])

(def event-names-enum
  [:enum "clinician_created" "clinician_updated" "patient_group_assignments_changed"
   "role_assignments_changed" "patient_created" "patient_updated"
   "patient_circle_of_care_created" "patient_circle_of_care_updated"
   "patient_circle_of_care_deleted" "threshold_created" "threshold_updated"
   "threshold_deleted" "role_created" "role_updated" "role_deleted"
   "patient_accessed" "patient_login" "clinician_login" "guardian_login"
   "patient_impersonated" "patient_metadata_changed"])

(def query-params-schema
  [:map
   [:event-name {:optional true} [:vector event-names-enum]]
   [:responsible-url {:optional true} [:re schemas/url-regex]]
   [:resource-url {:optional true} [:re schemas/url-regex]]
   [:resource-type {:optional true} string?]
   [:where {:optional true} [:re schemas/ip-address-regex]]
   [:offset {:optional true} int?]
   [:max {:optional true} int?]
   [:from {:optional true} [:re schemas/iso-8601-regex]]
   [:to {:optional true} [:re schemas/iso-8601-regex]]])

(m/=> get-audit-events [:=>
                        [:cat schemas/opts-schema query-params-schema]
                        [:or schemas/error-schema (schemas/success-schema events-schema)]])

(defn ^:mockable get-audit-events
  "Retrieve a list of audit events."
  [opts params]
  (let [url (str (:base-url opts) "/audit/events")
        res (client/get url
                        (assoc (utils/opts->request opts)
                               :multi-param-style :array
                               :query-params params))
        post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                    (f events-schema data utils/prune-map-transformer)))]
    (utils/->output [200] post-processor res)))

(defn ^:mockable get-audit-event
  "Retrieve a single audit event"
  [opts event-id]
  (let [url (format "%s/audit/events/%s" (:base-url opts) event-id)
        res (client/get url
                        (utils/opts->request opts))
        post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                    (f event-schema data utils/prune-map-transformer)))]
    (utils/->output [200] post-processor res)))

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope #{:input}
                 :report utils/input-validate-fail!})
(clj-kondo/emit!)
 ;; Enable mocks
(utils/make-mockable)

(comment
  ;; Example usage
  (get-audit-events {:base-url "https://doccla-dev.oth.io"
                     :validate-output? true
                     :auth {:type :id-secret :id "" :secret ""}}
                    {:event-name ["patient_created" "patient_updated"] :max 3})

  (get-audit-event {:base-url "https://doccla-dev.oth.io"
                    :validate-output? true
                    :auth {:type :id-secret :id "" :secret ""}}
                   "c3c8c303-65f0-4173-bac0-17b165daaa99"))
