(ns doccla.oth-client.clinician.api.patients.measurements
  (:require
   [clj-http.client :as client]
   [doccla.oth-client.utils :as utils]
   [malli.clj-kondo :as clj-kondo]
   [malli.core :as m]
   [malli.instrument :as mi]
   [doccla.oth-client.schemas :as schemas]))

(def query-params-schema
  [:map
   [:type {:optional true} [:enum schemas/measurement-types-enum]]
   [:ignored {:optional true} boolean?]
   [:isBeforeMeal {:optional true} boolean?]
   [:isAfterMeal {:optional true} boolean?]
   [:isFasting {:optional true} boolean?]
   [:offset {:optional true} int?]
   [:max {:optional true} int?]
   [:from {:optional true} [:re schemas/iso-8601-regex]]
   [:to {:optional true} [:re schemas/iso-8601-regex]]])

(m/=> get-measurements [:=>
                        [:cat schemas/opts-schema [:or string? int?] query-params-schema]
                        [:or schemas/error-schema (schemas/success-schema schemas/measurements-schema)]])

(defn ^:mockable get-measurements
  "Retrieve a list of measurements for a given patient id"
  [opts patient_id params]
  (let [url (str (:base-url opts) "/clinician/api/patients/" patient_id "/measurements")
        res (client/get url
                        (assoc (utils/opts->request opts)
                               :multi-param-style :array
                               :query-params params))
        post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                    (f schemas/measurements-schema data utils/prune-map-transformer)))]
    (utils/->output [200] post-processor res)))

(m/=> get-measurement [:=>
                       [:cat schemas/opts-schema [:or string? int?] [:or string? int?]]
                       [:or schemas/error-schema (schemas/success-schema schemas/measurement-schema)]])

(defn ^:mockable get-measurement-by-link
  "Retrieve a given measurement by its url"
  [opts url]
  (let [res (client/get url (utils/opts->request opts))
        post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                    (f schemas/measurement-schema data utils/prune-map-transformer)))]
    (utils/->output [200] post-processor res)))

(defn ^:mockable get-measurement
  "Retrieve a given measurement"
  [opts patient-id measurement-id]
  (let [url  (str (:base-url opts) "/clinician/api/patients/" patient-id "/questionnaire-results/" measurement-id)]
    (get-measurement-by-link opts url)))

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope #{:input}
                 :report utils/input-validate-fail!})
(clj-kondo/emit!)
 ;; Enable mocks
(utils/make-mockable)

(comment
  ;; Example usage
  (get-measurements {:base-url "https://doccla-dev.oth.io"
                     :validate-output? true
                     :auth {:type :id-secret :id "" :secret ""}}
                    37
                    {:max 3})

  (get-measurement-by-link {:base-url "https://doccla-dev.oth.io"
                            :validate-output? true
                            :auth {:type :id-secret :id "xxx" :secret "xxx"}}
                           "https://doccla-dev.oth.io/clinician/api/patients/22/measurements/29927"))
