(ns doccla.oth-client.clinician.api.patients.measurements
  (:require
   [clj-http.client :as client]
   [doccla.oth-client.utils :as utils]
   [malli.clj-kondo :as clj-kondo]
   [malli.core :as m]
   [malli.instrument :as mi]
   [doccla.oth-client.schemas :as schemas]))

(def blood-pressure-schema
  [:map
   [:timestamp [:re schemas/iso-8601-regex]]
   [:type [:enum "blood_pressure"]]
   [:measurement [:map [:systolic number?]
                  [:diastolic number?]
                  [:unit [:enum "mmHg"]]]]])

(def blood-sugar-schema
  [:map
   [:timestamp [:re schemas/iso-8601-regex]]
   [:type [:enum "bloodsugar"]]
   [:measurement [:map [:unit [:enum "mmol/L"]]
                  [:value number?]
                  [:is-after-meal {:optional true} boolean?]
                  [:is-before-meal {:optional true} boolean?]
                  [:is-fasting {:optional true} boolean?]
                  [:is-control-measurement {:optional true} boolean?]
                  [:ignored {:optional true} map?]]]])

(def blood-sugar-mg-dl
  [:map
   [:timestamp [:re schemas/iso-8601-regex]]
   [:type [:enum "bloodsugar_mg_dl"]]
   [:measurement [:map [:unit [:enum "mg/dL"]]
                  [:value number?]
                  [:is-after-meal {:optional true} boolean?]
                  [:is-before-meal {:optional true} boolean?]
                  [:is-fasting {:optional true} boolean?]
                  [:is-control-measurement {:optional true} boolean?]
                  [:ignored {:optional true} map?]]]])

(def measurement-types-simple-enum
  [:enum  "news2" "consciousness" "phase_angle" "fat_mass" "body_cell_mass" "weight" "weight_pound" "pulse" "temperature"
   "temperature_fahrenheit" "pain_scale" "blood_in_urine" "protein_in_urine" "leukocytes_in_urine"
   "glucose_in_urine" "nitrite_in_urine" "hemoglobin" "saturation" "crp" "height" "daily_steps"
   "daily_steps_weekly_average" "respiratory_rate" "peak_flow" "sit_to_stand" "fev1" "fev6"
   "fev1/fev6" "fef25-75%" "fev1%" "fev6%" "copd_prediction" "oxygen_flow"])

(def simple-value-schema
  [:map
   [:timestamp [:re schemas/iso-8601-regex]]
   [:type measurement-types-simple-enum]
   [:measurement [:map [:unit [:enum nil? "kg" "BPM" "mmol/L" "°C" "°F" "g/dL" "%" "mg/L" "mL" "mV" "L" "mmHg" "-" "cm" "RR" "L/min" "L/s" "min" "hour" "step count" "repetitions" "°" "lb"]]
                  [:value number?]
                  [:ignored {:optional true} map?]]]])

(def urine-value-schema
  [:map
   [:timestamp [:re schemas/iso-8601-regex]]
   [:type measurement-types-simple-enum]
   [:measurement [:map [:unit [:enum "Ery/μL" "g/L" "Leu/µL" "mmol/L" "-"]]
                  [:value string?]
                  [:ignored {:optional true} map?]]]])

(def measurement-origin-schema
  [:map
   [:manual-measurement {:optional true} [:map [:entered-by [:enum "citizen" "clinician"]]]]
   [:calculated-measurement {:optional true} [:map [:calculated-by string?]]]
   [:device-measurement {:optional true} map?]])

(def triggered-threshold-schema
  [:map
   [:type {:optional true} string?]
   [:result {:optional true} string?]
   [:measurement-type {:optional true} string?]
   [:links {:optional true} [:map
                             [:threshold [:re schemas/url-regex]]]]])
(def measurements-schema
  [:map
   [:results [:vector [:and
                       [:or
                        blood-pressure-schema
                        blood-sugar-schema
                        blood-sugar-mg-dl
                        simple-value-schema
                        urine-value-schema]

                       [:map
                        [:severity {:optional true} string?]
                        [:triggered-thresholds {:optional true} [:vector triggered-threshold-schema]]
                        [:comment {:optional true} string?]
                        [:origin {:optional true} measurement-origin-schema]
                        [:links [:map
                                 [:measurement [:re schemas/url-regex]]
                                 [:patient {:optional true} [:re schemas/url-regex]]]]]]]]
   [:offset int?]
   [:max int?]
   [:total int?]
   [:links [:map
            [:self [:re schemas/url-regex]]
            [:previous {:optional true} [:re schemas/url-regex]]
            [:next {:optional true} [:re schemas/url-regex]]]]])

(def query-params-schema
  [:map
   [:type {:optional true} [:enum schemas/measurement-types-enum]]
   [:ignored {:optional true} boolean?]
   [:isBeforeMeal {:optional true} boolean?]
   [:isAfterMeal {:optional true} boolean?]
   [:isFasting {:optional true} boolean?]
   [:offset {:optional true} int?]
   [:max {:optional true} int?]
   [:from {:optional true} [:re schemas/iso-8601-regex]]
   [:to {:optional true} [:re schemas/iso-8601-regex]]])

(m/=> get-measurements [:=>
                        [:cat schemas/opts-schema [:or string? int?] query-params-schema]
                        [:or schemas/error-schema (schemas/success-schema measurements-schema)]])

(defn ^:mockable get-measurements
  "Retrieve a list of measurements for a given patient id"
  [opts patient_id params]
  (let [url (str (:base-url opts) "/clinician/api/patients/" patient_id "/measurements")
        res (client/get url
                        (assoc (utils/opts->request opts)
                               :multi-param-style :array
                               :query-params params))
        post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                    (f measurements-schema data utils/prune-map-transformer)))]
    (utils/->output [200] post-processor res)))

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope #{:input}})
(clj-kondo/emit!)
 ;; Enable mocks
(utils/make-mockable)

(comment
  ;; Example usage
  (get-measurements {:base-url "https://doccla-dev.oth.io"
                     :validate-output? true
                     :auth {:type :id-secret :id "" :secret ""}}
                    37
                    {:max 3}))
