(ns doccla.oth-client.clinician.api.clinicians
  (:require
   [clj-http.client :as client]
   [doccla.oth-client.schemas :as schemas]
   [doccla.oth-client.utils :as utils]
   [malli.clj-kondo :as clj-kondo]
   [malli.core :as m]
   [malli.instrument :as mi]
   [malli.util :as mu]))

(def patient-group-schema
  [:map [:name string?] [:links [:map [:patient-group string?]]]])

(def role-schema
  [:map [:name string?] [:links [:map [:role string?]]]])

(def clinician-schema-base
  [:map
   {:closed true}
   [:unique-id {:optional true} string?]
   [:first-name string?]
   [:last-name string?]
   [:email {:optional true} string?]
   [:username string?]
   [:phone {:optional true} string?]
   [:mobile-phone {:optional true} string?]])

(def clinician-schema-read
  (mu/merge
   clinician-schema-base
   [:map
    [:patient-groups
     [:vector patient-group-schema]]
    [:roles
     [:vector role-schema]]
    [:links
     [:map
      [:self string?]]]]))

(def clinician-schema-create
  (mu/merge clinician-schema-base
            [:map
             [:links [:map
                      {:closed true}
                      [:patient-groups  [:vector {:min 1}
                                         [:re {:error/message "Must be a valid URL"} schemas/url-regex]]]
                      [:roles           [:vector {:min 1}
                                         [:re {:error/message "Must be a valid URL"} schemas/url-regex]]]]]]))

(def clinician-schema-update [:map])

(def clinician-schema-with-temporary-password
  (mu/merge clinician-schema-read
            [:map
             [:temporary-password string?]]))

(m/=> create-clinician [:=>
                        [:cat schemas/opts-schema clinician-schema-create]
                        [:or schemas/error-schema (schemas/success-schema clinician-schema-with-temporary-password)]])

(defn ^:mockable create-clinician
  "Create a new clinician."
  [opts clinician]
  (let [res (client/post (str (:base-url opts) "/clinician/api/clinicians")
                         (merge (utils/opts->request opts)
                                {:body (utils/encode clinician)}))
        post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                    (f clinician-schema-with-temporary-password data utils/prune-map-transformer)))]
    (utils/->output [201] post-processor res)))

(m/=> get-clinician [:=>
                     [:cat schemas/opts-schema [:or :string :int]]
                     [:or schemas/error-schema (schemas/success-schema clinician-schema-read)]])

(defn ^:mockable get-clinician
  "Get a clinician by ID."
  [opts clinician-id]
  (let [res (client/get (str (:base-url opts) "/clinician/api/clinicians/" clinician-id)
                        (utils/opts->request opts))
        post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                    (f clinician-schema-read data utils/prune-map-transformer)))]
    (utils/->output [200] post-processor res)))

(m/=> update-clinician [:=>
                        [:cat schemas/opts-schema [:or :string :int] clinician-schema-update]
                        [:or schemas/error-schema (schemas/success-schema clinician-schema-read)]])

(defn ^:mockable update-clinician
  "Update a clinician."
  [opts clinician-id clinician]
  (let [res (client/put (str (:base-url opts) "/clinician/api/clinicians/" clinician-id)
                        (merge (utils/opts->request opts)
                               {:body (utils/encode clinician)}))
        post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                    (f clinician-schema-read data utils/prune-map-transformer)))]
    (utils/->output [200] post-processor res)))

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope #{:input}})
(clj-kondo/emit!)
;; Enable mocks
(utils/make-mockable)

(comment
  (require '[clojure.edn :as edn])
  (def creds (edn/read-string (slurp "creds/oth.edn")))
  (def opts {:base-url    "https://doccla-dev.oth.io"
             :validate-output? true
             :auth {:type :id-secret :id (:id creds) :secret (:secret creds)}})
  (create-clinician
   opts
   {:first-name  "John"
    :last-name   "Doe"
    :username    "JohnDoe102"
    :links       {:patient-groups
                  ["https://doccla-dev.oth.io/clinician/api/patientgroups/27"]
                  :roles
                  ["https://doccla-dev.oth.io/clinician/api/roles/1"]}
    :unique-id   "JohnDoe102"}))