(ns doccla.oth-client.utils
  (:require
   [camel-snake-kebab.core :as csk]
   [clojure.test :as ct]
   [jsonista.core :as json]
   [malli.core :as m]
   [malli.instrument :as mi]
   [malli.transform :as mt]
   [clojure.pprint :refer [pprint]]
   [doccla.oth-client.schemas :as schemas]) (:import (clojure.lang ExceptionInfo)))

(defn prune-map-transformer []
  (mt/transformer
   {:decoders {:map #(into {} (filter (fn [[_k v]] (not (nil? v))) %))}}))

(defn decode [json-string]
  (json/read-value json-string (json/object-mapper {:decode-key-fn csk/->kebab-case-keyword})))

(defn encode [clj-data]
  (json/write-value-as-string clj-data (json/object-mapper {:encode-key-fn csk/->camelCaseString})))

(defn ->output
  "Creates a success or error map.
  If provided, will run the post-processor over the decoded input"
  ([success-codes input] (->output success-codes identity input))
  ([success-codes post-processor input]
   (if (some #{(:status input)} success-codes)
     {:code (:status input)
      :success? true
      :response (-> (:body input) decode (#(try
                                             (post-processor %)
                                             (catch ExceptionInfo e
                                               (when (= :malli.core/invalid-input (-> e ex-data :type))
                                                 (pprint (ex-data e)))
                                               (throw e)))))}
     {:code (:status input)
      :success? false
      :error (:body input)})))

(def default-request-map
  {:as               :json-kebab-keys
   :accept           :json
   :throw-exceptions false})

(m/=> opts->request [:=> [:cat schemas/opts-schema] :map])
(defn opts->request
  "Creates a clj-http request map from the opts passed in."
  [opts]
  (let [auth (:auth opts)]
    (cond-> default-request-map
      (= (:type auth) :token)     (assoc :headers {:authorization (:token auth)})
      (= (:type auth) :id-secret) (assoc :basic-auth [(:id auth) (:secret auth)]))))

(defn opts->request-with-body
  "Creates a clj-http POST request map from the given opts and body"
  [opts body]
  (-> (opts->request opts)
      (assoc :headers {:content-type "application/json"})
      (assoc :body (encode body))))

(comment
  (opts->request {:base-url "http" :validate-output? false :auth {:type :token, :token "foo"}})
  (opts->request {:base-url "http" :validate-output? false :auth {:type :id-secret :id "foo" :secret "bar"}}))

(defn- mock [opts fn-name args]
  (let [impl (get-in opts [:mocks fn-name])
        state (:mocks-state opts)]
    (swap! state #(assoc % fn-name {:args args}))
    (cond
      (ct/function? impl) (apply impl opts args)
      (nil? impl) {}                    ;TODO: What should the default mock be?
      :else
      impl)))

(defn make-mockable
  "Replaces any functions marked :mockable in the current namespace with an
  implementation that supports returning mock values."
  []
  (let [mockables (filter #(-> % second meta :mockable) (ns-publics *ns*))]
    (doseq [[fn-name fn-symbol] mockables]
      (alter-var-root fn-symbol (fn [f] (fn [opts & args]
                                          (if (:mock opts)
                                            (mock opts fn-name args)
                                            (apply f opts args))))))))

(m/=> ->mock-client [:=> [:cat [:map-of :symbol :any]] schemas/opts-schema])
(defn ->mock-client
  "Creates a client that, when passed to any oth-client functions,
  will cause them to return mock data as defined in mocks.
  `mocks` is a map of symbols to either data or functions. If a function
  is passed, that function is executed with the same args as the mocked function.
  If data is passed, the data is returned when the mock is called.
  Note that enabling mocks disables malli validation of inputs.
  "
  [mocks]
  {:base-url "http://test.com", :auth {:type :token :token "test-token"}
   :validate-output? false
   :mock true
   :mocks mocks
   :mocks-state (atom {})})

(mi/instrument!
 {:filters [mi/-filter-ns *ns*]})
