(ns doccla.oth-client.pagination)

(defn paginate-results
  "Paginates through results by sequentially calling a provided function `f` with an increasing offset,
  to fetch a specified total number of results. The function `f` should accept an offset parameter and return
  a map with keys `:response`, containing a nested map with keys `:total`, `:max`, and `:results`.

  The `number-of-results` parameter determines the total number of results to fetch. If it is not provided or
  if it is greater than the total number of available results, all results are fetched.

  This function will handle the pagination logic by injecting the offset parameter to the provided function `f`,
  so it is necessary to provide an offset parameter to `f` and add it in the query parameters.

  If one of the calls to `f` fails, the response of `f` will be returned immediately, without any further calls to `f`.

  Usage example:
  (paginate-results
    (fn [offset]
      (get-audit-events oth-client {:event-name [\"patient_updated\"] :offset offset :max 10}))
    100)

  In this example, `paginate-results` is used to fetch 100 audit events,
  calling `get-audit-events` with batches of 10 results each time.

  The return value will be in the same shape of the response of `f`,
  with the offset, max and links keys removed as they become irrelevant when the response has been accumulated, e.g.
  {:code     200
   :success? true
   :response {:total 100
              :results [{:some \"data\"} ...]}}
  "
  [f & [number-of-results]]
  (let [initial-res (f 0)]
    (if (not (:success? initial-res))
      initial-res ; Return initial response if not successful
      (let [total (get-in initial-res [:response :total])
            max-per-call (get-in initial-res [:response :max])
            number-of-results (Math/min (or number-of-results total) total)
            times-to-call (int (Math/ceil (/ number-of-results (double max-per-call))))]
        (if (<= times-to-call 1)
          initial-res ; Only one page of results needed
          (let [final-response (reduce (fn [accumulated-res n]
                                         (let [offset (* n max-per-call)
                                               res (f offset)]
                                           (if (:success? res)
                                             (update-in accumulated-res [:response :results] into (get-in res [:response :results]))
                                             (reduced res))))
                                       initial-res
                                       (range 1 times-to-call))]
            (update-in final-response [:response] dissoc :offset :max :links)))))))
