(ns doccla.oth-client.clinician.api.patient-notes
  (:require
   [clj-http.client :as client]
   [doccla.oth-client.utils :as utils]
   [doccla.oth-client.schemas :as schemas]
   [malli.clj-kondo :as clj-kondo]
   [malli.core :as m]
   [malli.instrument :as mi]))

(def patient-schema
  [:map
   {:closed true}
   [:username string?]
   [:unique-id string?]
   [:first-name string?]
   [:last-name string?]
   [:links [:map
            [:patient string?]]]])

(def clinician-summary-schema
  [:map
   {:closed true}
   [:email {:optional true} string?]
   [:first-name {:optional true} string?]
   [:last-name {:optional true} string?]
   [:username {:optional true} string?]
   [:links [:map [:clinician string?]]]])

(def note-schema
  [:map
   {:closed true}
   [:note string?]
   [:type [:enum "normal" "important"]]
   [:reminder-date {:optional true} string?]
   [:patient patient-schema]
   [:created-date string?]
   [:created-by {:optional true} clinician-summary-schema]
   [:read-by [:vector clinician-summary-schema]]
   [:links [:map
            [:patient-note string?]]]])

(def notes-schema
  [:map
   {:closed true}
   [:results [:vector note-schema]]
   [:total int?]
   [:max int?]
   [:offset int?]
   [:links [:map [:self string?]]]])

(def params-schema
  [:map
   [:patient {:optional true} string?]
   [:sort {:optional true} [:enum "createdBy" "note" "type" "reminderDate"]]
   [:order {:optional true} [:enum "desc" "asc"]]
   [:from {:optional true} string?]
   [:to {:optional true} string?]
   [:max {:optional true} int?]])

(m/=> get-notes [:=>
                 [:cat schemas/opts-schema params-schema]
                 [:or schemas/error-schema (schemas/success-schema notes-schema)]])

(defn ^:mockable get-notes
  "Retrieve a set of notes."
  [opts params]
  (let [res (client/get (str (:base-url opts) "/clinician/api/patient-notes/")
                        (assoc (utils/opts->request opts)
                               :query-params params))
        post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                    (f notes-schema data utils/prune-map-transformer)))]
    (utils/->output [200] post-processor res)))

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope #{:input}})
(clj-kondo/emit!)
;; Enable mocks
(utils/make-mockable)

(comment
  (get-notes {:base-url "https://doccla-dev.oth.io" :validate-output? true :auth {:type :id-secret :id "" :secret ""}} {}))
