(ns doccla.oth-client.clinician.api.patients.patients
  (:require
   [clj-http.client :as client]
   [doccla.oth-client.schemas :as schemas]
   [doccla.oth-client.utils :as utils]
   [malli.clj-kondo :as clj-kondo]
   [malli.core :as m]
   [malli.instrument :as mi]
   [malli.util :as mu]))

(def patient-group-schema
  [:map [:name string?] [:links [:map [:patient-group string?]]]])

(def relative-schema-base
  [:map
   {:closed true}
   [:first-name string?]
   [:last-name string?]
   [:relation string?]
   [:phone string?]
   [:address {:optional true} string?]
   [:city {:optional true} string?]
   [:note {:optional true} string?]
   [:email {:optional true} string?]
   [:guardian {:optional true}
    [:map
     [:username string?]
     [:email {:optional true} string?]
     [:unique-id {:optional true} string?]]]])

(def relative-schema-read
  (mu/merge
   relative-schema-base
   [:map
    [:unique-id {:optional true} string?]
    [:username {:optional true} string?]
    [:links [:map [:relative string?]]]]))

(def relative-schema-create relative-schema-base)

(def patient-schema-base
  [:map
   {:closed true}
   [:unique-id string?]
   [:first-name string?]
   [:last-name string?]
   [:address string?]
   [:postal-code string?]
   [:city string?]
   [:place {:optional true} string?]
   [:phone {:optional true} string?]
   [:mobile-phone {:optional true} string?]
   [:email {:optional true} string?]
   [:profile-picture {:optional true} [:re schemas/base-64-regex]]
   [:sex [:enum "male" "female" "unknown"]]
   [:due-date {:optional true} [:re schemas/simple-date-regex]]
   [:username string?]
   [:date-of-birth {:optional true} [:re schemas/simple-date-regex]]
   [:comment {:optional true} string?]])

(def patient-status-schema
  [:enum "active" "deceased" "discharged" "discharged_self_managed" "paused"])

(def patient-schema-read
  (mu/merge
   patient-schema-base
   [:map
    [:created-date string?]
    [:status patient-status-schema]
    [:relatives [:vector relative-schema-read]]
    [:patient-groups
     [:vector patient-group-schema]]
    [:links
     [:map
      [:measurements string?]
      [:questionnaire-results string?]
      [:contact-info string?]
      [:questionnaire-schedules string?]
      [:notifications string?]
      [:thresholds string?]
      [:questionnaires string?]
      [:self string?]
      [:change-password string?]]]]))

(m/=> get-patient [:=>
                   [:cat schemas/opts-schema [:or :string :int]]
                   [:or schemas/error-schema (schemas/success-schema patient-schema-read)]])

(defn get-patient-id-by-url [url]
  (second (re-matches #".*/patients/(\d+)" url)))

(defn ^:mockable get-patient
  "Retrieve a specific patient."
  [opts id]
  (let [res (client/get (str (:base-url opts) "/clinician/api/patients/" id)
                        (utils/opts->request opts))
        post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                    (f patient-schema-read data utils/prune-map-transformer)))]
    (utils/->output [200] post-processor res)))

;;;
;;; Create patient
;;; 

(def patient-schema-create
  (mu/merge patient-schema-base
            [:map
             [:relatives [:vector relative-schema-create]]
             [:links [:map
                      {:closed true}
                      [:patient-groups  [:vector {:min 1}
                                         [:re {:error/message "Must be a valid URL"} schemas/url-regex]]]
                      [:data-responsible {:optional true}
                       [:re {:error/message "Must be a valid URL"} schemas/url-regex]]]]]))

(def patient-schema-create-resp
  (mu/merge patient-schema-read
            [:map
             [:relatives
              [:vector (mu/assoc relative-schema-read
                                 [:temporary-password {:optional true}] string?)]]
             [:temporary-password {:optional true} string?]]))

(m/=> create-patient [:=>
                      [:cat schemas/opts-schema patient-schema-create]
                      [:or schemas/error-schema (schemas/success-schema patient-schema-create-resp)]])

(defn ^:mockable create-patient
  "Create a new patient."
  [opts patient]
  (let [res (client/post (str (:base-url opts) "/clinician/api/patients/")
                         (merge (utils/opts->request opts)
                                {:body (utils/encode patient)}))
        post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                    (f patient-schema-create-resp data utils/prune-map-transformer)))]
    (utils/->output [201] post-processor res)))

(def patient-schema-update
  (-> patient-schema-create
      (mu/assoc :status patient-status-schema)
      (mu/optional-keys)))

(defn insert-updates
  "Given a patient response (i.e. a map conforming to patent-schema-read)
   and a set of updates (i.e. a map conforming to patient-schema-update)
   returns a new map that satisfies the patient/{id}/PUT endpoint."
  [patient updates]
  (-> patient
      ;; move the patient groups to links
      ;; (the get enpoint returns a map of patient groups, but
      ;; the put endpoint expects a list of links) 
      (assoc :links {:patient-groups (vec (map (comp :patient-group :links)
                                               (:patient-groups patient)))})
      (dissoc :patient-groups)
      ;; remove any other un-needed keys for update
      (dissoc :created-date)
      ;; then merge in any updates. 
      (merge updates)))

(m/=> update-patient [:=>
                      [:cat schemas/opts-schema [:or :string :int] patient-schema-update]
                      [:or schemas/error-schema (schemas/success-schema patient-schema-create-resp)]])
(defn ^:mockable update-patient
  "Update an existing patient. Differs slightly from the OTH API in that
   it supports partial updates rather than wholesale replacement of every value.
   N.B. Patient groups and relatives will be replaced wholesale if provided."
  [opts id updates]
  (let [{:keys [success? response] :as get-resp} (get-patient opts id)]
    (if success?
      (let [_ (tap> response)
            res (client/put (str (:base-url opts) "/clinician/api/patients/" id)
                            (merge (utils/opts->request opts)
                                   {:body (-> (insert-updates response updates)
                                              (utils/encode))}))
            post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                        (f patient-schema-create-resp data utils/prune-map-transformer)))]
        (utils/->output [200] post-processor res))

      ;; If the get-patient step failed, return that failure as the response
      get-resp)))

;;;
;;; Common
;;;

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope #{:input}})
(clj-kondo/emit!)
;; Enable mocks
(utils/make-mockable)

(comment
  (require '[clojure.edn :as edn])
  (def creds (edn/read-string (slurp "creds/oth.edn")))
  (def opts {:base-url    "https://doccla-dev.oth.io"
             :validate-output? true
             :auth {:type :id-secret :id (:id creds) :secret (:secret creds)}})

  (get-patient (utils/->mock-client {'get-patient "foo"}) 1)
  (def p1 (:response (get-patient opts 1)))

  (create-patient
   opts
   {:first-name  "John"
    :last-name   "Doe"
    :username    "JohnDoe102"
    :sex         "male"
    :postal-code "123"
    :address     "abc"
    :city        "ny"
    :links       {:patient-groups
                  ["https://doccla-dev.oth.io/clinician/api/patientgroups/27"]}
    :relatives   []
    :unique-id   "JohnDoe102"})

  (update-patient opts 1 {:sex "male"}))