(ns doccla.oth-client.thresholds.patient-group-thresholds
  (:require [clj-http.client :as client]
            [doccla.oth-client.utils :as utils]
            [malli.clj-kondo :as clj-kondo]
            [malli.core :as m]
            [malli.instrument :as mi]
            [doccla.oth-client.schemas :as schemas]))

(def patient-group-thresholds-schema
  [:map
   [:total int?]
   [:max int?]
   [:offset int?]
   [:results
    [:vector [:and schemas/threshold-schema
              [:map [:links
                     [:map
                      [:threshold-absolute [:re utils/url-regex]]
                      [:patient-group [:re utils/url-regex]]
                      [:patient-group-threshold [:re utils/url-regex]]]]]]]]])

(def query-params-schema
  [:map
   [:patientGroup {:optional true} [:re utils/url-regex]]
   [:measurementType {:optional true} [:enum schemas/measurement-types-enum]]
   [:offset {:optional true} int?]
   [:max {:optional true} int?]])

(m/=> get-patient-group-thresholds [:=>
                                    [:cat utils/opts-schema query-params-schema]
                                    [:or utils/error-schema (utils/success-schema patient-group-thresholds-schema)]])

(defn post-processor
  [opts]
  (fn [data]
    (let [f (if (:validate-output? opts) m/coerce m/decode)]
      (f patient-group-thresholds-schema data utils/prune-map-transformer))))

(defn ^:mockable get-patient-group-thresholds
  "Retrieve the thresholds for a specific patient group"
  [opts params]
  (let [url (->> "/thresholds/patient-group-thresholds"
                 (str (:base-url opts)))
        req (-> (utils/opts->request opts)
                (assoc :query-params params
                       :multi-param-style :array))
        res (client/get url req)]
    (utils/->output [200] (post-processor opts) res)))

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope #{:input}})
;; Configure linting
(clj-kondo/emit!)
;; Enable mocks
(utils/make-mockable)

(comment
  ;; Example invocation
  (get-patient-group-thresholds {:base-url "https://doccla-dev.oth.io"
                                 :validate-output? true
                                 :auth {:type :id-secret
                                        :id ""
                                        :secret ""}}
                                {}
                                #_{:patientGroup "https://doccla-dev.oth.io/clinician/api/patientgroups/59"}))
