(ns doccla.oth-client.patients
  (:require
   [clj-http.client :as client]
   [doccla.oth-client.utils :as utils]
   [malli.clj-kondo :as clj-kondo]
   [malli.core :as m]
   [malli.instrument :as mi]))

(def patient-group-schema
  [:map [:name string?] [:links [:map [:patient-group string?]]]])

(def relative-schema
  [:map
   {:closed true}
   [:first-name string?]
   [:last-name string?]
   [:relation string?]
   [:phone string?]
   [:address {:optional true} string?]
   [:city {:optional true} string?]
   [:note {:optional true} string?]
   [:email {:optional true} string?]
   [:unique-id {:optional true} string?]
   [:is-guardian {:optional true} boolean?]
   [:username {:optional true} string?]
   [:links [:map [:relative string?]]]])

(def patient-schema
  [:map
   {:closed true}
   [:created-date string?]
   [:unique-id string?]
   [:first-name string?]
   [:last-name string?]
   [:address string?]
   [:postal-code string?]
   [:city string?]
   [:place {:optional true} string?]
   [:phone {:optional true} string?]
   [:mobile-phone {:optional true} string?]
   [:email {:optional true} string?]
   [:profile-picture {:optional true} [:re #"data:image.+;base64,"]]
   [:sex [:enum "male" "female" "unknown"]]
   [:username string?]
   [:patient-groups
    [:vector patient-group-schema]]
   [:date-of-birth {:optional true} [:re #"^(\d{4})[\-](0[1-9]|1[0-2])[\-]([12]\d|0[1-9]|3[01])$"]]
   [:status [:enum "active" "deceased" "discharged" "discharged_self_managed" "paused"]]
   [:comment {:optional true} string?]
   [:relatives
    [:vector relative-schema]]
   [:links
    [:map
     [:measurements string?]
     [:questionnaire-results string?]
     [:contact-info string?]
     [:questionnaire-schedules string?]
     [:notifications string?]
     [:thresholds string?]
     [:questionnaires string?]
     [:self string?]
     [:change-password string?]]]])

(m/=> get-patient [:=>
                   [:cat utils/opts-schema [:or :string :int]]
                   [:or utils/error-schema (utils/success-schema patient-schema)]])

(defn get-patient-id-by-url [url]
  (second (re-matches #".*/patients/(\d+)" url)))

(defn ^:mockable get-patient
  "Retrieve a specific patient."
  [opts id]
  (let [res (client/get (str (:base-url opts) "/clinician/api/patients/" id)
                        (utils/opts->request opts))
        post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                    (f patient-schema data utils/prune-map-transformer)))]
    (utils/->output [200] post-processor res)))

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(mi/-filter-ns *ns*)]
                 :scope #{:input}})
(clj-kondo/emit!)
;; Enable mocks
(utils/make-mockable)

(comment
  (get-patient (utils/->mock-client {'get-patient "foo"}) 1)
  (get-patient {:base-url "https://doccla-dev.oth.io" :validate-output? false :auth {:type :id-secret :id "xxx" :secret "yyy"}} 1))
