(ns doccla.oth-client.results.results
  (:require
   [clj-http.client :as client]
   [doccla.oth-client.utils :as utils]
   [malli.clj-kondo :as clj-kondo]
   [malli.core :as m]
   [malli.instrument :as mi]))

(def severity-enum [:enum "none" "green" "yellow" "orange" "red"])
(def processing-status-enum [:enum "pending" "ok" "error" "timeout"])
(def source-enum [:enum "external" "consultation" "conference" "questionnaire"])

(def threshold-explanation-schema
  [:map
   [:type [:enum "absolute" "relative-period" "relative-latest" "segment-lag" nil?]]
   [:result [:enum "undeterminable" "warning" "alert" "within-limits" "no-threshold"]]
   [:aggregate [:map
                [:count int?]
                [:percentage number?]]]
   [:links [:map
            [:threshold [:re utils/url-regex]]]]])

(def measurements-schema
  [:vector {:min-count 1}
   [:map
    [:timestamp [:re utils/iso-8601-regex]]
    [:type string?]
    [:severity {:optional true} severity-enum]
    [:comment {:optional true} string?]
    [:triggered-thresholds {:optional true} [:vector threshold-explanation-schema]]
    [:measurement {:optional true} map?]
    [:measurement-status {:optional true}
     [:map
      [:status [:enum "pending" "ready" "failed-will-retry" "failed-measurement-lost"]]]]]])

(def capture-summary-schema
  [:map
   [:result-time string?]
   [:processing-status processing-status-enum]
   [:source source-enum]
   [:acknowledged {:optional true} boolean?]
   [:measurements measurements-schema]
   [:result-type {:optional true} [:enum "capture"]]
   [:severity {:optional true} severity-enum]
   [:processed-at {:optional true} string?]
   [:ignored {:optional true} boolean?]
   [:links {:optional true} [:map
                             [:patient [:re utils/url-regex]]
                             [:result [:re utils/url-regex]]
                             [:measurement-capture [:re utils/url-regex]]]]])

(def input-reply-schema
  [:map
   [:reply-type [:enum "inputReply"]]
   [:reply [:map
            [:severity severity-enum]
            [:answer [:map
                      [:type [:enum "boolean" "number" "integer" "string"]]
                      [:value [:or boolean? number? string? nil?]]]]]]])

(def multiple-choice-sum-question-reply-schema
  [:map
   [:reply-type [:enum "multipleChoiceSumQuestionReply"]]
   [:reply [:map
            [:severity severity-enum]
            [:answer [:map
                      [:value number?]
                      [:text string?]]]]]])

(def multiple-choice-sum-reply-schema
  [:map
   [:reply-type [:enum "multipleChoiceSumReply"]]
   [:reply [:map
            [:severity severity-enum]
            [:answer [:map
                      [:value number?]
                      [:type [:enum "integer"]]]]]]])

(def multiple-choice-reply-schema
  [:map
   [:reply-type [:enum "multipleChoiceReply"]]
   [:reply [:map
            [:severity severity-enum]
            [:answer [:map
                      [:value string?]
                      [:text string?]]]]]])

(def omitted-reply-schema
  [:map
   [:reply-type [:enum "omittedReply"]]
   [:reply [:map
            [:omitted [:enum true]]
            [:severity severity-enum]]]])

(def measurement-reply-summary-schema
  [:map
   [:reply-type [:enum "measurementReply"]]
   [:reply capture-summary-schema]])

(def capture-reply-summary-schema
  [:map
   [:reply-type [:enum "captureReply"]]
   [:reply capture-summary-schema]])

(def questions-and-replies-summary-schema
  [:vector
   [:and
    [:map
     [:text string?]
     [:patient-text string?]
     [:reply-id [:re utils/url-regex]]]
    [:or input-reply-schema
     multiple-choice-sum-question-reply-schema
     multiple-choice-sum-reply-schema
     multiple-choice-reply-schema
     omitted-reply-schema
     measurement-reply-summary-schema
     capture-reply-summary-schema]]])

(def questionnaire-summary-schema
  [:map
   [:result-type [:enum "questionnaire"]]
   [:acknowledged boolean?]
   [:result-time [:re utils/iso-8601-regex]]
   [:processing-status processing-status-enum]
   [:severity {:optional true}  severity-enum]
   [:processed-at {:optional true} [:re utils/iso-8601-regex]]
   [:questions {:optional true} questions-and-replies-summary-schema]
   [:questionnaire-name {:optional true} string?]
   [:links [:map
            [:patient [:re utils/url-regex]]
            [:result [:re utils/url-regex]]
            [:questionnaire [:re utils/url-regex]]
            [:questionnaire-result [:re utils/url-regex]]]]])

(def results-schema
  [:map
   [:total int?]
   [:max int?]
   [:offset int?]
   [:results [:vector [:or capture-summary-schema
                       questionnaire-summary-schema]]]
   [:links [:map
            [:self [:re utils/url-regex]]
            [:next {:optional true} [:re utils/url-regex]]
            [:previous {:optional true} [:re utils/url-regex]]]]])

(def query-params-schema
  [:map
   [:patient {:optional true} [:re utils/url-regex]]
   [:questionnaire {:optional true} string?]
   [:acknowledged {:optional true} boolean?]
   [:resultType {:optional true} [:enum "capture" "questionnaire"]]
   [:offset {:optional true} int?]
   [:max {:optional true} int?]
   [:from {:optional true} [:re utils/iso-8601-regex]]
   [:to {:optional true} [:re utils/iso-8601-regex]]
   [:order {:optional true} [:enum "desc" "asc"]]])

(m/=> get-results [:=>
                   [:cat utils/opts-schema query-params-schema]
                   [:or utils/error-schema (utils/success-schema results-schema)]])

(defn ^:mockable get-results
  "Retrieve a list of results."
  [opts params]
  (let [res (client/get (str (:base-url opts) "/results/results")
                        (assoc (utils/opts->request opts)
                               :query-params params))
        post-processor (fn [data] (let [f (if (:validate-output? opts) m/coerce m/decode)]
                                    (f results-schema data utils/prune-map-transformer)))]
    (utils/->output [200] post-processor res)))

;; Enable instrumentation so library users get schema checking.
(mi/instrument! {:filters [(-> *ns* str symbol mi/-filter-ns)]
                 :scope #{:input}})
(clj-kondo/emit!)
;; Enable mocks
(utils/make-mockable)

(comment
  ;; Example usage
  (get-results {:base-url "https://doccla-dev.oth.io"
                :validate-output? true
                :auth {:type :id-secret :id "" :secret ""}}
               {:resultType "capture" :max 300}))
