(ns ^:no-doc com.dmo-t.ipaddress.impl.helpers
  (:require
   [clojure.math :refer [log]]
   [clojure.spec.alpha :as spec]
   [clojure.string :as str]
   [com.dmo-t.ipaddress.impl.constants :as constants]
   [com.dmo-t.ipaddress.specs :as isp])
  (:import
   [java.net  Inet6Address]))


(def ^:dynamic *compressed* true)

(defn ipv4addr-str->int
  [ip]
  (reduce (fn [acc v]
            (+ (bit-shift-left acc 8)  v)) (->> (str/split ip #"\.")
                                                (map parse-long))))

(defn int->ipv4addr-str
  [l]
  (loop [result '() remaining l count 0]
    (if (= 4 count)
      (str/join "." result)
      (recur (conj result (bit-and remaining 2r11111111))
             (bit-shift-right remaining 8)
             (inc count)))))

(comment
  (-> (ipv4addr-str->int "255.255.255.0"))

  constants/IPv4-ALL-ONES
  :rcf)


(defn v4-prefixlen->int [bl]
  (let [bl (if (string? bl) (parse-long bl)  bl)]
    (reduce (fn [acc n]
              (bit-clear acc n)) constants/IPv4-ALL-ONES (range 0  (- 32 bl)))))

(comment
  (ipv4addr-str->int "255.255.255.253")
  (-> (v4-prefixlen->int 31)
      int->ipv4addr-str)
  :rcf)

(defn netmask->hostmask
  [netmask]
  (->
   (.to-int netmask)
   Long/toBinaryString
   (str/replace "1" "")
   (str/replace "0" "1")
   (Long/parseUnsignedLong 2)))

(defn netmaskv4-int->prefixlen [netmask-int]
  (->> netmask-int
       Long/toBinaryString
       (filter #(= \1 %))
       count))

(defn ivp6-split-zoneid
  [full-addr]
  (let [splitted (str/split full-addr #"\%")]
    (if (< 2 (count splitted))
      (throw (Exception. "IPv6 address cannot have multiple zone separator ('%')"))
      splitted)))

(defn string->inet6address
  [addr]
  (Inet6Address/getByName addr))

(def xf-ipv6-to-binstr
  (comp
   (map #(Byte/toUnsignedInt %))
   (map #(Integer/toBinaryString %))
   (map #(let [c (count %)
               n (- 8 c)]
           (if (zero? n)
             %
             (str (-> (repeat n "0") str/join) %))))))


(defn inet6->binstr [^Inet6Address ipv6]
  (let [bb (.getAddress ipv6)]
    (-> (transduce xf-ipv6-to-binstr conj [] bb)
        str/join)))

(defn binstr->bigint [binstr]
  (->   (BigInteger. binstr  2)
        bigint))

(defn inet6address->bigint [ip]
  (-> (inet6->binstr ip)
      binstr->bigint))




(defn- compress-address-info [coll]
  (reduce (fn [{:keys [best-start best-length current-start] :as acc} [idx v]]
            (let [acc (cond-> acc
                        (and (= "0" v) (nil? current-start))
                        (-> (assoc :current-start idx)
                            (update :current-length inc))
                        (and (= "0" v) current-start)
                        (update :current-length inc)
                        (not= "0" v)
                        (-> (assoc :current-start nil)
                            (assoc :current-length 0)))]
              (cond-> acc
                (and (:current-start acc) (nil? best-start))
                (-> (assoc :best-start (:current-start acc))
                    (assoc :best-length (:current-length acc)))
                (and (:current-start acc) (< best-length (:current-length acc)))
                (-> (assoc :best-start (:current-start acc))
                    (assoc :best-length (:current-length acc))))))
          {:current-start nil
           :current-length 0
           :best-start nil
           :best-length 0} (map-indexed #(-> [%1 %2]) coll)))

(defn- set-empty-string [{:keys [best-start best-length]} coll]
  (let [coll (vec coll)
        l (count coll)]
    (cond
      (nil? best-start) coll
      (and (zero? best-start) (= (dec l) (dec (+ best-start best-length))))
      ["" "" ""]
      (zero? best-start)
      (into ["" ""] (subvec coll (+ best-start best-length)))
      (= (dec l) (dec (+ best-start best-length)))
      (into (subvec coll 0 best-start) ["" ""])
      :else
      (let [end-index (+ best-start best-length)]
        (concat (subvec coll 0 best-start)
                [""]
                (subvec coll end-index))))))

(defn ipv6str->compressed-ipv6str [ipv6-str]
  (if (re-matches #".*::.*" "ipv6-str")
    ipv6-str
    (let [coll (vec (str/split ipv6-str #":"))]
      (cond->> (set-empty-string (compress-address-info coll)
                                 coll)
        true (str/join ":")))))

(defn- format-hextet [i]
  (->> i
       (map #(Integer/toHexString %))
       (map #(if (< (count %) 2)
               (str "0" %)
               %))
       reverse
       (partition 2)
       (map str/join)
       (map #(str/replace % #"^0+" "0"))
       (map #(str/replace % #"^0+(.+)$" "$1"))
       (str/join ":")))

(defn bigint->ipv6str [i]
  (-> (loop [result [] remaining (biginteger i) counter 0]
        (if (= 16 counter)
          result
          (recur (conj result (BigInteger/.mod (biginteger remaining) (biginteger 256)))
                 (.shiftRight remaining 8)
                 (inc counter))))
      format-hextet
      (cond->
       *compressed* ipv6str->compressed-ipv6str)))



(comment
  (let [ip1 (string->inet6address "1080:0:0:0:8:800:200C:417A")
        ip2 (string->inet6address "::7A")
        ip3 (string->inet6address "2001:658:22a:cafe::")
        ip4 (string->inet6address "0:0:3:0:0:0:0:ffff")
        ip5 (string->inet6address "0:0:0:0:0:0:0:0")]
    (->  (inet6address->bigint ip3)
         bigint->ipv6str))
  :rcf)

(defn v6-prefixlen->bigint [bl]
  (let [bl (if (string? bl) (parse-long bl)  bl)]
    (-> (reduce (fn [acc n]
                  (.clearBit acc n)) (-> (string->inet6address constants/IPv6-ALL-ONES)
                                         inet6address->bigint
                                         biginteger) (range 0  (- 128 bl)))
        bigint)))

(comment

  (-> (v6-prefixlen->bigint 125)
      bigint->ipv6str)
  :rcf)

(defn bigint->prefixlen [netmask-int]
  (-> netmask-int
      biginteger
      (.toString 2)
      (as-> $ (filter #(= \1 %) $))
      count))

(defmulti to-binary-string class)
(defmethod to-binary-string Long
  [l]
  (Long/toBinaryString l))
(defmethod to-binary-string clojure.lang.BigInt
  [l]
  (-> (biginteger l)
      to-binary-string))
(defmethod to-binary-string java.math.BigInteger
  [l]
  (.toString l 2))

(defn bit-length [i]
  (if (zero? i)
    0
    (-> (log i)
        (/ (log 2))
        int)))

(comment
  (bit-length 260)
  :rcf)