(ns com.dmo-t.ipaddress.core
  (:require
   [clojure.math.numeric-tower :refer [expt]]
   [clojure.spec.alpha :as spec]
   [clojure.string :as str]
   [clojure.test :refer [is]]
   [com.dmo-t.ipaddress.impl.constants :as constants :refer [MAX_BITS]]
   [com.dmo-t.ipaddress.impl.helpers :as helpers]
   [com.dmo-t.ipaddress.specs :as isp])
  (:import
   [java.net  Inet6Address]))

(set! *warn-on-reflection* false)

(declare ->ipv4address ->ipv4network ->ipv6address ->ipv6network)
(def ^:no-doc IPv4-PRIVATE-NETWORKS (delay (mapv (fn [v] (->ipv4network v)) constants/IPv4-PRIVATE-NETWORKS)))

(def ^:no-doc IPv4-PRIVATE-NETWORKS-EXCEPTIONS (delay (mapv (fn [v] (->ipv4network v)) constants/IPv4-PRIVATE-NETWORKS-EXCEPTIONS)))
(def ^:no-doc IPv4-RESERVED-NETWORK (delay (->ipv4network constants/IPv4-RESERVED-NETWORK)))
(def ^:no-doc IPv4-UNSPECIFIED-ADDRESS (delay (->ipv4address constants/IPv4-UNSPECIFIED-ADDRESS)))
(def ^:no-doc IPv4-LINK-LOCAL-NETWORK (delay (->ipv4network constants/IPv4-LINK-LOCAL-NETWORK)))
(def ^:no-doc IPv4-LOOPBACK-NETWORK (delay (->ipv4network constants/IPv4-LOOPBACK-NETWORK)))
(def ^:no-doc IPv4-MULTICAST-NETWORK (delay (->ipv4network constants/IPv4-MULTICAST-NETWORK)))
(def ^:no-doc IPv4-PUBLIC-NETWORK (delay (->ipv4network constants/IPv4-PUBLIC-NETWORK)))

(def ^:no-doc IPv6-ALL-ONES (delay (-> (Inet6Address/getByName constants/IPv6-ALL-ONES)
                                       helpers/inet6address->bigint)))
(def ^:no-doc IPv6-LINK-LOCAL-NETWORK (delay (->ipv6network constants/IPv6-LINK-LOCAL-NETWORK)))
(def ^:no-doc IPv6-MULTICAST-NETWORK (delay (->ipv6network constants/IPv6-MULTICAST-NETWORK)))
(def ^:no-doc IPv6-PRIVATE-NETWORKS (delay (mapv ->ipv6network constants/IPv6-PRIVATE-NETWORKS)))
(def ^:no-doc IPv6-PRIVATE-NETWORKS-EXCEPTIONS (delay (mapv ->ipv6network constants/IPv6-PRIVATE-NETWORKS-EXCEPTIONS)))
(def ^:no-doc IPv6-RESERVED-NETWORKS (delay (mapv ->ipv6network constants/IPv6-RESERVED-NETWORKS)))
(def ^:no-doc IPv6-SITE-LOCAL-NETWORK (delay (->ipv6network constants/IPv6-SITE-LOCAL-NETWORK)))


(comment
  @IPv6-PRIVATE-NETWORKS

  :rcf)

(defn- scopeid-str [ip]
  (let [scope (.scopeid ip)]
    (if (zero? scope)
      ""
      (str "%" scope))))

(defn- =?* [this other]
  (zero? (compare this other)))

(defn- <?* [this other]
  (neg? (compare this other)))

(defn- >?* [this other]
  (pos? (compare this other)))

(defn- <=?* [this other]
  (or
   (<?* this other)
   (=?* this other)))
(defn- >=?* [this other]
  (or
   (>?* this other)
   (=?* this other)))


(defn- equiv-test [f this other]
  (when-not (= (class this) (class other))
    (throw (ex-info "other must be of same class than this" {:this-class (class this)
                                                             :other-class (class other)})))
  (f this other))

(defprotocol IIPIter
  (next-one
    [this]
    "Returns the next entry of same type.
     - IPv[46]Address: returns the next IPv[46]Address
     - IPv[46]Network: returns the next IPv[46]Network with same netmask,
     - IPv[46]Interface: returns the next IPv[46]Interface in the same network.
     Returns nil when no more entry is available.
     (next-one nil) returns nil
     "))



(defprotocol IIPBase
  (toString [this]
    "Returns the default string representation of the type.
     - IPv[46]Address and IPv[46]Netmask: the address representation,
     - IPv[46]Network, IPv[46]Interface: address/prefixlen
     'str' calls this function.
     ")
  (version [this]
    "Returns the IPAddress version: 4 or 6.")
  (max-prefixlen [this]
    "Returns the maximum value of prefix length (size in bits of IP object)")
  (=? [this other]
    "Returns (zero? (compare this other))")
  (<? [this other]
    "Returns (neg? (compare this other))")
  (>? [this other]
    "Returns (pos? (compare this other))")
  (<=? [this other]
    "Returns (or (=? this other) (<? this other)")
  (>=? [this other]
    "Returns (or (=? this other) (>? this other)"))

(defprotocol IIPAddrConv
  (to-int [this]
    "Convert an IPv[46]Address to int"))
(defprotocol IIPAddress
  (in? [this net]
    "Returns true if the IPv[46]Address is contained in IPv[46]Network"))

(defprotocol IIPScope
  "Various predicates, testing the address, network type/scope."
  (loopback? [this])
  (link-local? [this])
  (multicast? [this])
  (private? [this])
  (global? [this])
  (reserved? [this])
  (unspecified? [this]))

(deftype IPv4Address [version* ip]
  IIPAddress
  (in? [this net]
    (.contains-ip?  net this))

  IIPBase
  (toString [_]
    (helpers/int->ipv4addr-str ip))
  (version [_]
    version*)
  (max-prefixlen [_]
    (MAX_BITS version*))
  (=? [this other]
    (equiv-test =?* this other))
  (<? [this other]
    (equiv-test <?* this other))
  (>? [this other]
    (equiv-test >?* this other))
  (<=? [this other]
    (equiv-test <=?* this other))
  (>=? [this other]
    (equiv-test >=?* this other))

  IIPAddrConv
  (to-int [_]
    ip)

  IIPIter
  (next-one [_]
    (if (= constants/IPv4-ALL-ONES ip)
      nil
      (IPv4Address. version* (inc ip))))

  Comparable
  (compareTo [_ other]
    (when-not (instance? IPv4Address other)
      (throw (ex-info "other must be an IPv4Address" {:msg "other must be an IPv4Address"
                                                      :other-class (class other)})))
    (cond
      (< ip (.ip other)) -1
      (<  (.ip other) ip) 1
      :else
      0))

  IIPScope
  (loopback? [this]
    (in? this @IPv4-LOOPBACK-NETWORK))
  (link-local? [this]
    (in? this @IPv4-LINK-LOCAL-NETWORK))
  (multicast? [this]
    (in? this @IPv4-MULTICAST-NETWORK))

  (private? [this]
    (if (some (partial in? this) @IPv4-PRIVATE-NETWORKS-EXCEPTIONS)
      false
      (-> (some (partial in? this) @IPv4-PRIVATE-NETWORKS)
          boolean)))
  (global? [this]
    (and (-> (private? this)
             not)
         (-> (.contains-ip? @IPv4-PUBLIC-NETWORK this))))
  (reserved? [this]
    (in? this @IPv4-RESERVED-NETWORK))
  (unspecified? [this]
    (-> (compare this @IPv4-UNSPECIFIED-ADDRESS)
        zero?)))

(defmulti ->ipv4address
  "Used to create and IPv4Address.
   'ip' can be a 'Long' or a 'String'."
  class)

(defmethod ->ipv4address Long
  [ip]
  (when-not (spec/valid? ::isp/ipv4-address-spec ip)
    (throw (ex-info (format "'%d' is not a valid integer for an IPv4 address." ip) {:msg (spec/explain-str ::isp/ipv4-address-spec ip)})))
  (->IPv4Address 4 ip))


(defmethod ->ipv4address String
  [ip]
  (when-not (spec/valid? ::isp/ipv4-address-spec ip)
    (throw (ex-info (format "'%s' is not a valid IPv4 address string." ip) {:msg (spec/explain-str ::isp/ipv4-address-spec ip)})))
  (-> (helpers/ipv4addr-str->int ip)
      ->ipv4address))

(defprotocol IIPNetmask
  (to-prefixlen [this]
    "Converts an IPv[46]Netmask to a prefixlen (that is an int)")
  (to-hostmask [this]
    "Converts an IPv[46]Netmask to an IPv[46]Address representing the hostmask"))

(deftype IPv4Netmask
         [netmask* prefixlen*]
  IIPAddrConv
  (to-int [_]
    (to-int netmask*))

  IIPBase
  (toString [_]
    (str netmask*))
  (version [_]
    (version netmask*))
  (max-prefixlen [this]
    (MAX_BITS (version this)))
  (=? [this other]
    (equiv-test =?* this other))
  (<? [this other]
    (equiv-test <?* this other))
  (>? [this other]
    (equiv-test >?* this other))
  (<=? [this other]
    (equiv-test <=?* this other))
  (>=? [this other]
    (equiv-test >=?* this other))


  IIPNetmask
  (to-prefixlen [_]
    prefixlen*)

  (to-hostmask [this]
    (-> (to-int this)
        Long/toBinaryString
        (str/replace "1" "")
        (str/replace "0" "1")
        (Long/parseUnsignedLong 2)
        ((fn [l] (->IPv4Address (version this) l))))))

(defmulti ^:private ->ipv4netmask
  "Creates an IPv4Netmask.
   Input can be an 'int' or a 'string'"
  class)
(defmethod ^:private ->ipv4netmask Long
  [netmask*]
  (when-not (spec/valid? ::isp/ipv4-netmask-spec netmask*)
    (throw (ex-info "Wrong input format. Must abide to to ipv4-netmask-spec" {:msg "Wrong input format. Must abide to to ipv4-netmask-spec"
                                                                              :calue netmask*})))
  (->IPv4Netmask (->ipv4address netmask*) (helpers/netmaskv4-int->prefixlen netmask*)))
(defmethod ^:private ->ipv4netmask String
  [netmask*]
  (-> (helpers/ipv4addr-str->int netmask*)
      ->ipv4netmask))

(comment
  (let [netmask (->ipv4netmask "255.255.254.0")]
    (class netmask))
  :rcf)

(defprotocol IIPStr
  (with-prefixlen [this]
    "Returns a string representation in the format: addr/prefixlen")
  (with-netmask [this]
    "Returns a string representation in the format: addr/netmask")
  (with-hostmask [this]
    "returns a string representation in the format: addr/hostmask"))


(defprotocol IIPNetwork
  (base-address [this]
    "(Legacy: use `network-address` instead). Returns the network address: IPv[46]Address type. ")
  (network-address [this]
    "Returns the network address: IPv[46]Address type. (Same base-address)")
  (netmask [this]
    "Returns the network netmask: IPv[46]Netmask type.")
  (broadcast-address [this]
    "Returns the network broadcast address: IPv[46]Address type.")
  (prefixlen [this]
    "Returns the network prefixlen: int")
  (hostmask [this]
    "Returns the network hostmask: IPv[46]Address")
  (hosts [this]
    "Returns the IPv[46]Address usable for host addressing in this network. Lazy seq.")
  (subnet-of? [this other]
    "Tests if the nerwork is a subnet of other network")
  (supernet-of? [this other]
    "Tests if the network is a superneet of other network.")
  (contains-ip? [this ip]
    "Tests if the network contains the IPv[46]Address")
  (overlaps? [this other]
    "Tests is the two IPv[46]Network overlap.")
  (subnets [this new-prefixlen]
    "Returns a list of IPv[46]Network with new-prefixlen which are subnets of this.")
  (supernet [this new-prefixlen]
    "Returns the supernet with new-prefixlen that contains the network.")
  (num-addresses [this]
    "Compute the number of addresses in the network.")
  (address-exclude [this other]
    "Returns a list of subnets of the network, with the exclusion of other")
  (to-seq [this]
    "Returns a lazy seq of all IPv[46]Address contained in the network."))

(defn- list-all-addresses* [net]
  (let [broadcast-address (broadcast-address net)]
    (->> (iterate next-one (base-address net))
         (take-while #(let [d (compare % broadcast-address)]
                        (or
                         (neg? d)
                         (zero? d)))))))


(defn- broadcast-address-fn* [net]
  (cond
    (and
     (= 4 (version net))
     (= (MAX_BITS 4) (prefixlen net)))
    (base-address net)
    (and
     (= 6 (version net))
     (= (MAX_BITS 6) (prefixlen net)))
    (base-address net)
    :else
    (let [version* (version net)
          network-int (to-int (base-address net))
          hostmask-int (-> (hostmask net)
                           to-int)
          b (+ network-int hostmask-int)]
      (if (= 4 version*)
        (->ipv4address b)
        (->ipv6address [b (.scopeid (base-address net))])))))

(defn- subnet-of?* [this other]
  (let [base-compare (compare (base-address this) (base-address other))
        broadcast-compare (compare (broadcast-address this) (broadcast-address other))]
    (and
     (or
      (pos? base-compare)
      (zero? base-compare))
     (or
      (neg? broadcast-compare)
      (zero? broadcast-compare)))))

(defn- contains-ip?* [this ip]
  (let [base-compare (compare (base-address this) ip)
        broadcast-compare (compare  (broadcast-address this) ip)]
    (and
     (or
      (neg? base-compare)
      (zero? base-compare))
     (or
      (pos? broadcast-compare)
      (zero? broadcast-compare)))))

(defn- overlaps?* [this other]
  (let [[f s] (sort-by identity #(compare %1 %2) [this other])]
    (->> (compare  (base-address  s) (broadcast-address f))
         ((some-fn neg? zero?)))))

(defn- num-addresses* [this]
  (let [base-address-int  (to-int (base-address this))
        broadcast-address-int (to-int (broadcast-address this))]
    (-> (- broadcast-address-int base-address-int)
        inc)))

(defn- address-exclude* [this other]
  (when (not (supernet-of? this other))
    (let [msg (format "%s not contained in %s" (with-prefixlen other) (with-prefixlen this))]
      (throw (ex-info msg {:msg msg}))))
  (if (zero? (compare this other))
    '()
    (let [new-prefixlen (inc (prefixlen this))
          [s1 s2] (subnets this new-prefixlen)]
      (cond
        (zero? (compare s1 other)) [s2]
        (zero? (compare s2 other)) [s1]
        (supernet-of? s1 other)
        (lazy-seq (cons s2  (address-exclude s1 other)))
        (supernet-of? s2 other)
        (lazy-seq (cons s1  (address-exclude s2 other)))   :else
        (throw (Exception. "Should not be there"))))))

(deftype IPv4Network [base-address* netmask*]
  IIPScope
  (loopback? [this]
    (subnet-of? this @IPv4-LOOPBACK-NETWORK))
  (link-local? [this]
    (subnet-of? this @IPv4-LINK-LOCAL-NETWORK))
  (multicast? [this]
    (subnet-of? this @IPv4-MULTICAST-NETWORK))

  (private? [this]
    (if (some (partial subnet-of? this) @IPv4-PRIVATE-NETWORKS-EXCEPTIONS)
      false
      (-> (some (partial subnet-of? this) @IPv4-PRIVATE-NETWORKS)
          boolean)))

  (global? [this]
    (and (-> (private? this)
             not)
         (-> (subnet-of?  this @IPv4-PUBLIC-NETWORK)
             not)))
  (reserved? [this]
    (subnet-of? this @IPv4-RESERVED-NETWORK))
  (unspecified? [this]
    (and
     (-> (compare base-address* @IPv4-UNSPECIFIED-ADDRESS)
         zero?)
     (= (MAX_BITS 4) (prefixlen this))))

  IIPNetwork
  (base-address [_]
    base-address*)
  (network-address [_]
    base-address*)
  (netmask [_]
    netmask*)
  (broadcast-address [this]
    (if (= (MAX_BITS 4) (prefixlen this))
      base-address*
      (let [network-int (to-int base-address*)
            hostmask-int (-> (hostmask this)
                             to-int)]
        (-> (+ network-int hostmask-int)
            ->ipv4address))))
  (prefixlen [_]
    (to-prefixlen netmask*))
  (hostmask [_]
    (to-hostmask netmask*))
  (hosts [this]
    (let [pf (prefixlen this)]
      (case  pf
        (32 31) (list-all-addresses* this)
        (-> (list-all-addresses* this)
            next
            butlast))))
  (to-seq [this]
    (list-all-addresses* this))

  (subnet-of? [this other]
    (when-not (instance? IPv4Network other)
      (throw (ex-info "other must be an IPv4Network" {:msg "other must be an IPv4Network" :other-class (class other)})))
    (subnet-of?* this other))

  (supernet-of? [this other]
    (when-not (instance? IPv4Network other)
      (throw (ex-info "other must be an IPv4Network" {:msg "other must be an IPv4Network" :other-class (class other)})))
    (subnet-of? other this))

  (contains-ip? [this ip]
    (when-not (instance? IPv4Address ip)
      (throw (ex-info "ip must be an IPv4Address" {:msg "other must be an IPv4Address" :ip-class (class ip)})))
    (contains-ip?* this ip))

  (overlaps? [this other]
    (when-not (instance? IPv4Network other)
      (throw (ex-info "other must be an IPv4Network" {:msg "other must be an IPv4Network" :other-class (class other)})))
    (overlaps?* this other))

  (subnets [this new-prefixlen]
    (when-not (and
               (spec/valid? ::isp/ipv4-prefixlen-spec new-prefixlen)
               (<= (.prefixlen this) new-prefixlen))
      (throw (ex-info "New prefixlen must be greater than current prefixlen and be a valid one" {:msg "New prefixlen must be greater than current prefixlen"
                                                                                                 :current (.prefixlen this)
                                                                                                 :new new-prefixlen})))

    (let [new-netmask (-> (helpers/v4-prefixlen->int new-prefixlen)
                          ->ipv4netmask)
          b (broadcast-address this)
          new-network (IPv4Network.  base-address* new-netmask)]
      (->> (iterate next-one new-network)
           (take-while #(-> (compare  (broadcast-address %) b)
                            pos?
                            not)))))

  (supernet [this new-prefixlen]
    (when-not (and
               (spec/valid? ::isp/ipv4-prefixlen-spec new-prefixlen)
               (<= new-prefixlen (.prefixlen this)))
      (throw (ex-info "New prefixlen must be lower than current one and be a valid value."
                      {:msg "New prefixlen must be lower than current one and be a valid value."
                       :current (.prefixlen this)
                       :new new-prefixlen})))

    (let [new-netmask (-> (helpers/v4-prefixlen->int new-prefixlen)
                          ->ipv4netmask)
          new-base-address (-> (bit-and (to-int base-address*) (to-int new-netmask))
                               ->ipv4address)
          new-network (IPv4Network.  new-base-address new-netmask)]
      (if (<=
           (-> (broadcast-address new-network) to-int)
           constants/IPv4-ALL-ONES)
        new-network
        nil)))
  (num-addresses [this]
    (num-addresses* this))
  (address-exclude [this other]
    (when-not (instance? IPv4Network other)
      (throw (ex-info "Other must be an IPv4Network" {:msg "Other must be an IPv4Network"
                                                      :other-class (class other)})))
    (address-exclude* this other))

  IIPIter
  (next-one [this]
    (let [size (inc (- (to-int (broadcast-address this)) (to-int (base-address this))))
          new-start (+ size (-> (base-address this) (to-int)))
          new-broadcast (dec (+ size new-start))]
      (if (<= new-broadcast constants/IPv4-ALL-ONES)
        (IPv4Network.  (IPv4Address. (version base-address*) new-start) netmask*)
        nil)))
  IIPBase
  (toString [this]
    (with-prefixlen this))
  (version [_]
    (version base-address*))
  (max-prefixlen [this]
    (MAX_BITS (version this)))
  (=? [this other]
    (equiv-test =?* this other))
  (<? [this other]
    (equiv-test <?* this other))
  (>? [this other]
    (equiv-test >?* this other))
  (<=? [this other]
    (equiv-test <=?* this other))
  (>=? [this other]
    (equiv-test >=?* this other))


  IIPStr
  (with-netmask [_]
    (str  base-address* "/" netmask*))
  (with-prefixlen [this]
    (str  base-address* "/" (prefixlen this)))
  (with-hostmask [this]
    (str  base-address* "/" (hostmask this)))

  Comparable
  (compareTo [this other]
    (when-not (instance? IPv4Network other)
      (throw (ex-info "Other must be an IPv4Network" {:msg "Other must be an IPv4Network"
                                                      :other-class (class other)})))

    (let [b1 base-address*
          b2 (base-address ^IPv4Network other)
          cmp (compare b1 b2)]
      (if (not (zero? cmp))
        cmp
        (compare (prefixlen this) (prefixlen other))))))

(defmulti ->ipv4network
  "Creates an IPv4Network.
     'arg' must be in form network/prefixlen or network/netmask"
  class)

(defmethod ->ipv4network String
  [arg]
  (when-not (and
             (= 2 (count (str/split arg #"/")))
             (spec/valid? ::isp/ipv4-address-spec (nth (str/split arg #"/")  0))
             (or
              (spec/valid? ::isp/ipv4-prefixlen-spec (nth (str/split arg #"/")  1))
              (spec/valid? ::isp/ipv4-netmask-spec (nth (str/split arg #"/")  1))))
    (throw (ex-info "Input is not a valid network representation" {:msg "Input is not a valid network representation" :input arg})))

  (let [[network-str netmask-str] (str/split arg #"/")
        netmask-int (if (spec/valid? ::isp/ipv4-prefixlen-spec netmask-str)
                      (helpers/v4-prefixlen->int netmask-str)
                      (helpers/ipv4addr-str->int netmask-str))
        network-int (bit-and (helpers/ipv4addr-str->int network-str)  netmask-int)]
    (->IPv4Network  (->ipv4address network-int) (->ipv4netmask netmask-int))))

(defmethod ->ipv4network clojure.lang.PersistentVector
  [[ip prefixlen]]
  (when-not (spec/valid? ::isp/ipv4-address-spec ip)
    (throw (ex-info "Invalid Address format" {:msg (spec/explain-str ::isp/ipv4-address-spec ip)})))
  (when-not (spec/valid? ::isp/ipv4-prefixlen-spec prefixlen)
    (throw (ex-info "Invalid prefixlen format" {:msg (spec/explain-str ::isp/ipv4-prefixlen-spec prefixlen)})))
  (let [netmask (helpers/v4-prefixlen->int prefixlen)
        network-int (bit-and ip netmask)]
    (->IPv4Network (->ipv4address network-int) (->ipv4netmask netmask))))

(comment
  (let [n (->ipv4network "192.0.0.0/22")]
    (time  (->> (.subnets n 27)
                (map with-prefixlen))))
  (.num-addresses (->ipv4network "1.2.3.0/24"))
  (->> (.address-exclude (->ipv4network "1.2.3.0/24") (->ipv4network "1.2.3.23/32"))
       (map #(.with-prefixlen %)))
  (->> (.subnets (->ipv4network "1.2.3.24/31") 32)
       (map #(.with-prefixlen %)))
  :rcf)

(comment
  (-> (->ipv4network "255.255.255.252/31")
      (.supernet 29)
      .with-prefixlen)

  (->> (->ipv4network "10.47.1.0/29")
       .hosts
       (map #(str %)))

  :rcf)


(defprotocol IIPInterface
  (network [this]
    "Returns the IPv[46]Network containing the IPv[46]Interface.")
  (address [this]
    "Returns the IPv[46]Address of the IPv[46]Interface."))

(deftype IPv4Interface [address* netmask*]
  IIPScope
  (loopback? [this]
    (-> (network this)
        .loopback?))
  (link-local? [this]
    (-> (network this)
        .link-local?))
  (multicast? [this]
    (-> (network this)
        .multicast?))

  (private? [this]
    (-> (network this)
        .private?))
  (global? [_]
    (-> address*
        .global?))
  (reserved? [this]
    (-> (network this)
        .reserved?))
  (unspecified? [this]
    (-> (network this)
        .unspecified?))

  IIPInterface
  (network [this]
    (->ipv4network (.with-prefixlen this)))
  (address [_]
    address*)

  Comparable
  (compareTo [this other]
    (when-not (instance? IPv4Interface other)
      (throw (ex-info "other must be an IPv4Interface" {:msg "other must be an IPv4Interface"
                                                        :other-class (class other)})))
    (let [addr-cmp (compare address* (address other))
          net-cmp (compare (network this) (network other))]
      (if (zero? net-cmp)
        addr-cmp
        net-cmp)))

  IIPIter
  (next-one [this]
    (let [ba (-> (network this) broadcast-address)]
      (if (zero? (compare address* ba))
        nil
        (IPv4Interface.  (next-one address*) netmask*))))

  IIPBase
  (toString [this]
    (with-prefixlen this))
  (version [_]
    (version address*))
  (max-prefixlen [this]
    (MAX_BITS (version this)))
  (=? [this other]
    (equiv-test =?* this other))
  (<? [this other]
    (equiv-test <?* this other))
  (>? [this other]
    (equiv-test >?* this other))
  (<=? [this other]
    (equiv-test <=?* this other))
  (>=? [this other]
    (equiv-test >=?* this other))



  IIPStr
  (with-prefixlen [_]
    (-> (IPv4Network. address* netmask*)
        .with-prefixlen))
  (with-netmask [_]
    (-> (IPv4Network.  address* netmask*)
        .with-netmask))
  (with-hostmask [_]
    (-> (IPv4Network.  address* netmask*)
        .with-hostmask)))

(defn ->ipv4interface
  "Creates an IPv4Interface.
   'intf' must in format: ip/netmask or ip/prefixlen "
  [intf]
  (when-not  (= 2 (count (str/split intf #"/")))
    (throw (ex-info "Wrong input format: netmask/or prefilex must be specified" {:msg "Wrong input format: netmask/or prefilex must be specified"})))
  (when-not      (spec/valid? ::isp/ipv4-address-spec (nth (str/split intf #"/")  0))
    (throw (ex-info "Wrong IPv4address format" {:msg (spec/explain-str ::isp/ipv4-address-spec (nth (str/split intf #"/")  0))})))
  (when-not (or
             (spec/valid? ::isp/ipv4-prefixlen-spec (nth (str/split intf #"/")  1))
             (spec/valid? ::isp/ipv4-netmask-spec (nth (str/split intf #"/")  1)))
    (throw (ex-info "After the '/': must be a string representation of netmask or prefixlen" {:msg "After the '/': must be a string representation of netmask or prefixlen"})))

  (let [[ip-str netmask-str] (str/split intf #"/")
        netmask-int (if (spec/valid? ::isp/ipv4-prefixlen-spec netmask-str)
                      (helpers/v4-prefixlen->int netmask-str)
                      (helpers/ipv4addr-str->int netmask-str))
        ip-addr (->ipv4address ip-str)
        netmask (->ipv4netmask netmask-int)]
    (->IPv4Interface  ip-addr netmask)))

(extend-type nil
  IIPIter
  (next-one [_]
    nil))


(comment
  (time
   (dotimes [_ 1000000] (->ipv4network "1.1.1.1/24")))
  (-> (->ipv4interface "192.0.2.7/29") next-one next-one)
  :rcf)

;;IPv6

(defprotocol IIPv6Scope
  (site-local? [this]))

(defprotocol IIPv6Specific
  (scopeid [this])
  (ipv4-mapped [this]
    "Returns the IPv4Address if it is an ipv4 mapped address. 
     Returns nil otherwise."))

(deftype IPv6Address [version* ip-int* scopeid*]
  IIPv6Specific
  (scopeid [_]
    scopeid*)
  (ipv4-mapped [_]
    (let [b (biginteger ip-int*)]
      (if (== 0xffff (.shiftRight b (MAX_BITS 4)))
        (-> (.and b (biginteger 0xffffffff))
            long
            ->ipv4address)
        nil)))


  IIPBase
  (toString [this]
    (let [scopestr (scopeid-str this)]
      (if-some [ipv4 (ipv4-mapped this)]
        (str "::ffff:" ipv4)
        (str (helpers/bigint->ipv6str ip-int*) scopestr))))
  (version [_]
    version*)
  (max-prefixlen [this]
    (MAX_BITS (version this)))
  (=? [this other]
    (equiv-test =?* this other))
  (<? [this other]
    (equiv-test <?* this other))
  (>? [this other]
    (equiv-test >?* this other))
  (<=? [this other]
    (equiv-test <=?* this other))
  (>=? [this other]
    (equiv-test >=?* this other))


  IIPScope
  (link-local? [this]
    (in? this @IPv6-LINK-LOCAL-NETWORK))
  (loopback? [_]
    (== ip-int* 1))
  (multicast? [this]
    (in? this @IPv6-MULTICAST-NETWORK))
  (private? [this]
    (if (some (partial in? this) @IPv6-PRIVATE-NETWORKS-EXCEPTIONS)
      false
      (-> (some (partial in? this) @IPv6-PRIVATE-NETWORKS)
          boolean)))
  (global? [this]
    (not (private? this)))
  (unspecified? [_]
    (== 0 ip-int*))
  (reserved? [this]
    (-> (some (partial in? this) @IPv6-RESERVED-NETWORKS)
        boolean))

  IIPv6Scope
  (site-local? [this]
    (in? this @IPv6-SITE-LOCAL-NETWORK))

  IIPAddrConv
  (to-int [_]
    ip-int*)

  IIPIter
  (next-one [_]
    (if (<= @IPv6-ALL-ONES ip-int*)
      nil
      (IPv6Address. version* (inc ip-int*) scopeid*)))

  Comparable
  (compareTo [_ other]
    (when-not (instance? IPv6Address other)
      (throw (ex-info "other must be an IPv6Address." {:msg "other must be an IPv6Address."
                                                       :other-class (class other)})))
    (compare ip-int* (.ip-int* other)))

  IIPAddress
  (in? [this net]
    (contains-ip? net this)))


(defmulti ->ipv6address
  "Used to create IPV6address.
   The input must be string or a clojure.lang.BigInt.
   "
  class)
(defmethod ->ipv6address clojure.lang.BigInt
  [ip]
  (->ipv6address (helpers/bigint->ipv6str ip)))
(defmethod ->ipv6address String
  [ip]
  (let [result (helpers/string->inet6address ip)]
    (if (instance? java.net.Inet6Address result)
      (->IPv6Address 6 (helpers/inet6address->bigint result) (.getScopeId result))
      (let [ipv4-int (-> (->ipv4address (.getHostAddress result)) .to-int bigint)
            prefix-int (-> (bit-shift-left 0xffff (MAX_BITS 4)) bigint)]
        (->IPv6Address 6 (+ ipv4-int prefix-int) 0)))))
(defmethod ->ipv6address clojure.lang.PersistentVector
  [[ip-int scope]]
  (->IPv6Address 6 ip-int scope))


(comment
  (-> (->ipv6address "1080::8:800:200C:417A%123") str)
  (-> (->ipv6address "1080::8:800:200C:417A%123") str)
  (-> (->ipv6address "1080::8:800:200C:417A%0") scopeid)
  (-> (->ipv6address "::FFFF:129.144.52.38")  str)
  (-> (->ipv6address "Ffff:Ffff:Ffff:Ffff:Ffff:Ffff:Ffff:Fff8") str)
  :rcf)

(comment
  ((juxt
    site-local?
    loopback?
    link-local?
    multicast?) (->ipv6address "1080::8:800:200C:417A"))
  @IPv6-ALL-ONES
  :rcf)

(deftype IPv6Netmask
         [netmask prefixlen*]
  IIPAddrConv
  (to-int [_]
    (to-int netmask))

  IIPBase
  (toString [_]
    (str netmask))
  (version [_]
    (version netmask))
  (max-prefixlen [this]
    (MAX_BITS (version this)))
  (=? [this other]
    (equiv-test =?* this other))
  (<? [this other]
    (equiv-test <?* this other))
  (>? [this other]
    (equiv-test >?* this other))
  (<=? [this other]
    (equiv-test <=?* this other))
  (>=? [this other]
    (equiv-test >=?* this other))


  IIPNetmask
  (to-prefixlen [_]
    prefixlen*)

  (to-hostmask [_]
    (-> netmask
        to-int
        biginteger
        (.toString 2)
        (str/replace "1" "")
        (str/replace "0" "1")
        helpers/binstr->bigint
        ((fn [l] (IPv6Address. 6 l 0))))))

(defmulti ^:private ->ipv6netmask
  "Create an IPv6Netmask from a prefixlen (if arg is a Long).
   If arg is clojure.lang.BigInt: treats it as an int representation of the netmask.
   "
  class)
(defmethod ^:private ->ipv6netmask clojure.lang.BigInt
  [mask]
  (->IPv6Netmask  (-> mask
                      helpers/bigint->ipv6str
                      ->ipv6address)   (helpers/bigint->prefixlen mask)))
(defmethod ^:private ->ipv6netmask Long
  [pl]
  (when-not (spec/valid? ::isp/ipv6-prefixlen-spec pl)
    (throw (ex-info "Wrong prefixlen format" {:msg (spec/explain-str ::isp/ipv6-prefixlen-spec pl)})))
  (->ipv6netmask (helpers/v6-prefixlen->bigint pl)))
(defmethod ^:private ->ipv6netmask java.math.BigInteger
  [mask]
  (->ipv6netmask (bigint mask)))

(comment
  (-> (helpers/v6-prefixlen->bigint 125)
      helpers/bigint->prefixlen)
  (-> (->ipv6netmask (helpers/v6-prefixlen->bigint 125)))
  (-> (helpers/v6-prefixlen->bigint 48)
      helpers/bigint->ipv6str
      ->ipv6address)
  :rcf)

(deftype IPv6Network [base-address* mask*]
  IIPBase
  (version [_]
    (version base-address*))
  (toString [this]
    (with-prefixlen this))
  (max-prefixlen [this]
    (MAX_BITS (version this)))
  (=? [this other]
    (equiv-test =?* this other))
  (<? [this other]
    (equiv-test <?* this other))
  (>? [this other]
    (equiv-test >?* this other))
  (<=? [this other]
    (equiv-test <=?* this other))
  (>=? [this other]
    (equiv-test >=?* this other))

  IIPNetwork
  (base-address [_]
    base-address*)
  (network-address [_]
    base-address*)
  (netmask [_]
    mask*)
  (broadcast-address [this]
    (broadcast-address-fn* this))
  (prefixlen [_]
    (to-prefixlen mask*))
  (hostmask [_]
    (to-hostmask mask*))
  (hosts [this]
    (case (prefixlen this)
      (127 128) (to-seq this)
      (-> (to-seq this)
          next)))
  (subnet-of? [this other]
    (when-not (instance? IPv6Network other)
      (throw (ex-info "other must be an IPv6Network" {:msg "other must be an IPv6Network"
                                                      :other-class (class other)})))
    (subnet-of?* this other))
  (supernet-of? [this other]
    (when-not (instance? IPv6Network other)
      (throw (ex-info "other must be an IPv6Network" {:msg "other must be an IPv6Network"
                                                      :other-class (class other)})))

    (subnet-of? other this))
  (contains-ip? [this ip]
    (when-not (instance? IPv6Address ip)
      (throw (ex-info "ip must be an IPv6Address" {:msg "ip must be an IPv6Address"
                                                   :ip-class (class ip)})))
    (contains-ip?* this ip))
  (overlaps? [this other]

    (when-not (instance? IPv6Network other)
      (throw (ex-info "other must be an IPv6Network" {:msg "other must be an IPv6Network"
                                                      :other-class (class other)})))
    (overlaps?* this other))

  (subnets [this new-prefixlen]
    (when-not (spec/valid? ::isp/ipv6-prefixlen-spec new-prefixlen)
      (throw (ex-info "Wrong prefixlen format" {:msg (spec/explain-str ::isp/ipv6-prefixlen-spec new-prefixlen)})))
    (when-not (<= (prefixlen this) new-prefixlen)
      (throw (ex-info "the new-prefixlen cannot be lower than current one" {:msg "the new-prefixlen cannot be lower than current one"})))
    (let [new-netmask (-> (helpers/v6-prefixlen->bigint new-prefixlen)
                          ->ipv6netmask)
          b (broadcast-address this)
          new-network (IPv6Network.  base-address* new-netmask)]
      (->> (iterate next-one new-network)
           (take-while #(-> (compare  (base-address %) b)
                            pos?
                            not)))))
  (supernet [this new-prefixlen]
    (when-not (spec/valid? ::isp/ipv6-prefixlen-spec new-prefixlen)
      (throw (ex-info "Wrong prefixlen format" {:msg (spec/explain-str ::isp/ipv6-prefixlen-spec new-prefixlen)})))
    (when-not (<= new-prefixlen (prefixlen this))
      (throw (ex-info "the new-prefixlen cannot be greater than current one" {:msg "the new-prefixlen cannot be greater than current one"})))


    (let [new-netmask (-> (helpers/v6-prefixlen->bigint new-prefixlen)
                          ->ipv6netmask)
          new-base-address (-> (.and (biginteger (to-int base-address*)) (biginteger (to-int new-netmask)))
                               bigint
                               ->ipv6address)
          new-network (IPv6Network.  new-base-address new-netmask)]
      (if (<=
           (-> (broadcast-address new-network) to-int)
           @IPv6-ALL-ONES)
        new-network
        nil)))


  (num-addresses [this]
    (num-addresses* this))
  (address-exclude [this other]
    {:pre [(is (instance? IPv6Network other))]}
    (address-exclude* this other))
  (to-seq [this]
    (list-all-addresses* this))
  IIPIter
  (next-one [this]
    (let [base-int (to-int ^IPv6Address base-address*)
          size (inc (- (to-int (broadcast-address ^IPv6Network this))  base-int))
          new-start (+ size base-int)
          new-broadcast (dec (+ size new-start))]
      (if (<= new-broadcast @IPv6-ALL-ONES)
        (IPv6Network.  (IPv6Address. (version ^IPv6Address base-address*) new-start  (scopeid ^IPv6Address base-address*)) mask*)
        nil)))
  Comparable
  (compareTo [this other]
    (when-not (instance? IPv6Network other)
      (throw (ex-info "other must be an IPv6Network" {:msg "other must be an IPv6Network"})))
    (let [cmp-addr (compare (base-address this) (base-address other))
          cmp-prefixlen (compare (prefixlen this) (prefixlen other))]
      (if (not (zero? cmp-addr))
        cmp-addr
        cmp-prefixlen)))

  IIPStr
  (with-prefixlen [_]
    (str  base-address* "/" (to-prefixlen mask*)))
  (with-netmask [_]
    (str  base-address* "/"  mask*))
  (with-hostmask [_]
    (str  base-address* "/"  (to-hostmask mask*)))

  IIPScope
  (link-local? [this]
    (subnet-of? this @IPv6-LINK-LOCAL-NETWORK))
  (loopback? [this]
    (and (== (MAX_BITS 6) (prefixlen this)) (== 1 (to-int base-address*))))
  (multicast? [this]
    (subnet-of? this @IPv6-MULTICAST-NETWORK))
  (private? [this]
    (if (some (partial subnet-of? this) @IPv6-PRIVATE-NETWORKS-EXCEPTIONS)
      false
      (-> (some (partial subnet-of? this) @IPv6-PRIVATE-NETWORKS)
          boolean)))
  (global? [this]
    (not (private? this)))
  (unspecified? [this]
    (and (unspecified? base-address*)
         (= (MAX_BITS 6) (prefixlen this))))
  (reserved? [this]
    (-> (some (partial subnet-of? this) @IPv6-RESERVED-NETWORKS)
        boolean))

  IIPv6Scope
  (site-local? [this]
    (subnet-of? this @IPv6-SITE-LOCAL-NETWORK)))

(defmulti ->ipv6network
  "Create and IPv6Network. 
     'net' must be a string in format addr/prefixlen"

  class)

(defmethod ->ipv6network String
  [net]
  (let [[addr pl] (str/split net #"/")
        pl (if pl (parse-long pl) (MAX_BITS 6))
        ip6 (->ipv6address addr)
        netmask-int (helpers/v6-prefixlen->bigint pl)
        addr-in (->  ip6
                     to-int)
        scope-str (scopeid-str ip6)]
    (->IPv6Network (->  (.and (biginteger addr-in) (biginteger netmask-int))
                        bigint
                        helpers/bigint->ipv6str
                        (str scope-str)
                        ->ipv6address) (->ipv6netmask netmask-int))))

(defmethod ->ipv6network clojure.lang.PersistentVector
  [[ip prefixlen]]
  (let [netmask-int (helpers/v6-prefixlen->bigint prefixlen)]
    (->IPv6Network (-> (.and (biginteger ip) (biginteger netmask-int))
                       bigint
                       helpers/bigint->ipv6str
                       ->ipv6address)  (->ipv6netmask netmask-int))))

(comment
  (let [net (->ipv6network "2001:658:22a:cafe::/123")]
    (->> (subnets net 126)
         (map str)))
  (-> (->ipv6network "2001::/23")
      (subnets 24)
      (as-> $ (map str $)))

  (-> (->ipv6network "2001::/64") with-prefixlen)

  (let [net1 (->ipv6network "2001::/23")
        net2 (->ipv6network "2001:3::/32")]
    (->> (address-exclude net1 net2)
         sort
         (map str)))

  (->ipv6network [5192296858534827628530496329220096N 16])

  :rcf)

(deftype IPv6Interface [address* netmask*]
  IIPScope
  (loopback? [this]
    (-> (network this)
        .loopback?))
  (link-local? [this]
    (-> (network this)
        .link-local?))
  (multicast? [this]
    (-> (network this)
        .multicast?))

  (private? [this]
    (-> (network this)
        .private?))
  (global? [_]
    (-> address*
        .global?))
  (reserved? [this]
    (-> (network this)
        .reserved?))
  (unspecified? [this]
    (-> (network this)
        .unspecified?))

  IIPv6Scope
  (site-local? [this]
    (-> (network this)
        .site-local?))


  IIPInterface
  (network [this]
    (->ipv6network (str this)))
  (address [_]
    address*)

  Comparable
  (compareTo [this other]
    (when-not (instance? IPv6Interface other)
      (throw (ex-info "other must be an IPv6Interface." {:msg "other must be an IPv6Interface."})))
    (let [addr-cmp (compare address* (address other))
          net-cmp (compare (network this) (network other))]
      (if (zero? net-cmp)
        addr-cmp
        net-cmp)))

  IIPIter
  (next-one [this]
    (let [ba (-> (network this) broadcast-address)]
      (if (zero? (compare address* ba))
        nil
        (IPv6Interface.  (next-one address*) netmask*))))

  IIPBase
  (toString [this]
    (with-prefixlen this))
  (version [_]
    (version address*))
  (max-prefixlen [this]
    (MAX_BITS (version this)))
  (=? [this other]
    (equiv-test =?* this other))
  (<? [this other]
    (equiv-test <?* this other))
  (>? [this other]
    (equiv-test >?* this other))
  (<=? [this other]
    (equiv-test <=?* this other))
  (>=? [this other]
    (equiv-test >=?* this other))


  IIPStr
  (with-prefixlen [_]
    (-> (IPv6Network. address* netmask*)
        .with-prefixlen))
  (with-netmask [_]
    (-> (IPv6Network.  address* netmask*)
        .with-netmask))
  (with-hostmask [_]
    (-> (IPv6Network.  address* netmask*)
        .with-hostmask)))

(defn ->ipv6interface
  "Create and IPv6Interface. 
   'net' must be a string in format addr/prefixlen"
  [net]
  (let [[addr pl] (str/split net #"/")
        pl (if pl (parse-long pl) (MAX_BITS 6))
        addr (->ipv6address addr)
        netmask (->ipv6netmask pl)]
    (->IPv6Interface addr netmask)))

(comment
  (let [intf1 (->ipv6interface "2001:658:22a:cafe:200:0:0:1/64")
        intf2 (->ipv6interface "2001:658:22a:cafe:200:0:0:1")]
    (str (network intf1)))
  :rcf)

;; Functions usage with multiple objects

(defn- remove-dup-networks [coll]
  (let [constructor (if (instance? IPv4Network (first coll)) ->ipv4network ->ipv6network)]
    (->> (map str coll)
         (into #{})
         (map constructor))))



(comment
  (let [net1 (->ipv4network "192.168.1.0/24")
        net2 (->ipv4network "192.168.1.0/24")
        net3 (->ipv4network "192.168.2.0/24")]
    (remove-dup-networks [net1 net3 net2]))
  :rcf)

(defn- collapse-addresses* [coll]
  (loop [subnets {}  remaining (seq coll)]
    (if-let [net (first remaining)]
      (let [l (.prefixlen net)
            supernet (.supernet net (dec l))
            not-same? #(not (zero? (compare %1 %2)))
            s (str supernet)]
        (cond
          (nil? (get subnets s))
          (recur (assoc subnets (str supernet) net) (next remaining))
          (not-same? (get subnets s) net)
          (recur (dissoc subnets s) (cons supernet (next remaining)))
          :else (recur subnets (next remaining))))
      (vals subnets))))


(defn- collapse-addresses-last-stage [coll]
  (loop [result []  remaining (seq coll)]
    (cond
      (empty? remaining) result
      (= 1 (count remaining)) (conj result (first remaining))
      :else
      (let [net (first remaining)]
        (if-not (pos? (compare (.broadcast-address (second remaining))  (.broadcast-address net)))
          (recur result (concat [net]  (nnext remaining)))
          (recur (conj result net) (next remaining)))))))

(defn collapse-addresses
  "Collapse a list of IPNetwork"
  [coll]
  (when-not (sequential? coll)
    (throw (ex-info "coll must be sequential (vector, list)" {:msg "coll must be sequential (vector, list)"})))
  (when-not (< (count (set (map #(class %) coll))) 2)
    (throw (ex-info "Cannot mix types" {:msg "Cannot mix types"
                                        :types (set (map #(class %) coll))})))
  (when-not (or
             (empty? coll)
             (instance? IPv4Network (first coll))
             (instance? IPv6Network (first coll)))
    (throw (ex-info "Supports only IPv4Network or IPv6Network" {:msg "Supports only IPv4Network or IPv6Network"
                                                                :type (class (first coll))})))

  (-> (remove-dup-networks coll)
      sort
      collapse-addresses*
      sort
      collapse-addresses-last-stage))


(defn- num-zero-r-bits [i]
  (->> (helpers/to-binary-string i)
       (re-find #"0+$")
       count))

(defn- summarize-address-range* [max-prefixlen ip1 ip2]
  (loop [result [] ip1 ip1]
    (if (< ip2 ip1)
      result
      (let [bl  (min
                 (helpers/bit-length (inc (- ip2 ip1)))
                 (num-zero-r-bits ip1))
            pf (- max-prefixlen bl)]
        (recur (conj result [ip1 pf]) (+ ip1 (expt 2 bl)))))))



(defn summarize-address-range
  "Given the first and last IP Address: returns a list of IP Network representing the range. "
  [ip1 ip2]
  (when-not (or
             (every? (partial instance? IPv4Address) [ip1 ip2])
             (every? (partial instance? IPv6Address) [ip1 ip2]))
    (throw (ex-info "Supports only IPv4Address and IPv6Address. ip1 and ip2 must be of same type." {:msg "Supports only IPv4Address and IPv6Address. ip1 and ip2 must be of same type."
                                                                                                    :types [(class ip1) (class ip2)]})))
  (when  (pos? (compare ip1 ip2))
    (throw (ex-info "ip1 cannot be greater than ip2" {:msg "ip1 cannot be greater than ip2"
                                                      :ip1 (str ip1)
                                                      :ip2 (str ip2)})))
  (let [constructor (if (= 4 (version ip1)) ->ipv4network ->ipv6network)
        max-prefixlen (constants/MAX_BITS (version ip1))]

    (->> (summarize-address-range* max-prefixlen (to-int ip1) (to-int ip2))
         (map constructor))))

(comment
  (summarize-address-range (->ipv4address "192.168.1.1") (->ipv4address "192.168.1.124"))
  (->ipv4network [232235784 29])

  (let [ip1-v6 (->ipv6address "1::")
        ip2-v6 (->ipv6address "1:ffff:ffff:ffff:ffff:ffff:ffff:ffff")]
    (summarize-address-range ip1-v6 ip2-v6))
  :rcf)

;; generic constructors
(defn ->ipaddress
  "Tries to create an IPv4Address, if unsucessful: tries IPv6Address."
  [ip]
  (try
    (->ipv4address ip)
    (catch Exception _
      (->ipv6address ip))))

(defn ->ipnetwork
  "Tries to create an IPv4Network, if unsucessful: tries IPv6Network."
  [net]
  (try
    (->ipv4network net)
    (catch Exception _
      (->ipv6network net))))

(defn ->ipinterface
  "Tries to create an IPv4Interface, if unsucessful: tries IPv6Interface."
  [net]
  (try
    (->ipv4interface net)
    (catch Exception _
      (->ipv6interface net))))


(comment
  (str (->ipaddress "192.168.1.0"))
  (str (->ipaddress "2001:cafE:1::1"))
  (str (->ipnetwork "192.0.2.0/24"))
  (str (->ipnetwork "2001:cafe::/64"))
  (str (->ipinterface "192.0.2.1/24"))
  (str (->ipinterface "2001:cafe::24/64"))


  :rcf)