(ns com.dmo-t.ipaddress.core
  (:require
   [clojure.spec.alpha :as spec]
   [clojure.string :as str]
   [com.dmo-t.ipaddress.impl.constants :as constants]
   [com.dmo-t.ipaddress.impl.helpers :as helpers]
   [com.dmo-t.ipaddress.specs :as isp])
  (:import [java.net  Inet6Address]))

(set! *warn-on-reflection* false)
(set! *assert* true)

(declare make-ipv4address make-ipv4network make-ipv6address make-ipv6network)
(def ^:no-doc IPv4-PRIVATE-NETWORKS (delay (mapv (fn [v] (make-ipv4network v)) constants/IPv4-PRIVATE-NETWORKS)))

(def ^:no-doc IPv4-PRIVATE-NETWORKS-EXCEPTIONS (delay (mapv (fn [v] (make-ipv4network v)) constants/IPv4-PRIVATE-NETWORKS-EXCEPTIONS)))
(def ^:no-doc IPv4-RESERVED-NETWORK (delay (make-ipv4network constants/IPv4-RESERVED-NETWORK)))
(def ^:no-doc IPv4-UNSPECIFIED-ADDRESS (delay (make-ipv4address constants/IPv4-UNSPECIFIED-ADDRESS)))
(def ^:no-doc IPv4-LINK-LOCAL-NETWORK (delay (make-ipv4network constants/IPv4-LINK-LOCAL-NETWORK)))
(def ^:no-doc IPv4-LOOPBACK-NETWORK (delay (make-ipv4network constants/IPv4-LOOPBACK-NETWORK)))
(def ^:no-doc IPv4-MULTICAST-NETWORK (delay (make-ipv4network constants/IPv4-MULTICAST-NETWORK)))
(def ^:no-doc IPv4-PUBLIC-NETWORK (delay (make-ipv4network constants/IPv4-PUBLIC-NETWORK)))

(def ^:no-doc IPv6-ALL-ONES (delay (-> (Inet6Address/getByName constants/IPv6-ALL-ONES)
                                       helpers/inet6address->bigint)))
(def ^:no-doc IPv6-LINK-LOCAL-NETWORK (delay (make-ipv6network constants/IPv6-LINK-LOCAL-NETWORK)))
(def ^:no-doc IPv6-MULTICAST-NETWORK (delay (make-ipv6network constants/IPv6-MULTICAST-NETWORK)))
(def ^:no-doc IPv6-PRIVATE-NETWORKS (delay (mapv make-ipv6network constants/IPv6-PRIVATE-NETWORKS)))
(def ^:no-doc IPv6-PRIVATE-NETWORKS-EXCEPTIONS (delay (mapv make-ipv6network constants/IPv6-PRIVATE-NETWORKS-EXCEPTIONS)))
(def ^:no-doc IPv6-RESERVED-NETWORKS (delay (mapv make-ipv6network constants/IPv6-RESERVED-NETWORKS)))
(def ^:no-doc IPv6-SITE-LOCAL-NETWORK (delay (make-ipv6network constants/IPv6-SITE-LOCAL-NETWORK)))


(comment
  @IPv6-PRIVATE-NETWORKS

  :rcf)

(defn- scopeid-str [ip]
  (let [scope (.scopeid ip)]
    (if (zero? scope)
      ""
      (str "%" scope))))


(defprotocol IPIter
  (next-one
    [this]
    "Returns the next entry of same type.
     - IPv4Address: returns the next IPv4Address
     - IPv4Network: returns the next IPv4Network with same netmask,
     - IPv4Interface: returns the next IPv4Interface in the same network.
     Returns nil when no more entry is available.
     (next-one nil) returns nil
     "))



(defprotocol IPBase
  (to-string [this]
    "Returns the default string representation of the type.
     - IPv[46]Address and IPv[46]Netmask: the address representation,
     - IPv[46]Network, IPv[46]Interface: address/prefixlen
     ")
  (toString [this]
    "eg to-string")
  (version [this]
    "Returns the IPAddress version: 4 or 6."))
(defprotocol IPAddrConv
  (to-int [this]
    "Convert an IPv[46]Address to int"))
(defprotocol IPAddress
  (in? [this net]
    "Returns true if the IPv[46]Address is contained in IPv[46]Network"))

(defprotocol IPScope
  (loopback? [this])
  (link-local? [this])
  (multicast? [this])
  (private? [this])
  (global? [this])
  (reserved? [this])
  (unspecified? [this]))

(deftype IPv4Address [version* ip]
  IPAddress
  (in? [this net]
    (.contains-ip?  net this))

  IPBase
  (to-string [_]
    (helpers/int->ipv4addr-str ip))
  (toString [this]
    (to-string this))
  (version [_]
    version*)

  IPAddrConv
  (to-int [_]
    ip)

  IPIter
  (next-one [_]
    (if (= constants/IPv4-ALL-ONES ip)
      nil
      (IPv4Address. version* (inc ip))))

  Comparable
  (compareTo [_ other]
    {:pre [instance? IPv4Address other]}
    (cond
      (< ip (.ip other)) -1
      (<  (.ip other) ip) 1
      :else
      0))

  IPScope
  (loopback? [this]
    (in? this @IPv4-LOOPBACK-NETWORK))
  (link-local? [this]
    (in? this @IPv4-LINK-LOCAL-NETWORK))
  (multicast? [this]
    (in? this @IPv4-MULTICAST-NETWORK))

  (private? [this]
    (if (some (partial in? this) @IPv4-PRIVATE-NETWORKS-EXCEPTIONS)
      false
      (-> (some (partial in? this) @IPv4-PRIVATE-NETWORKS)
          boolean)))
  (global? [this]
    (and (-> (private? this)
             not)
         (-> (.contains-ip? @IPv4-PUBLIC-NETWORK this))))
  (reserved? [this]
    (in? this @IPv4-RESERVED-NETWORK))
  (unspecified? [this]
    (-> (compare this @IPv4-UNSPECIFIED-ADDRESS)
        zero?)))

(defn- make-ipv4address-from-int
  [ip]
  {:pre [(<= 0 ip) (<= ip constants/IPv4-ALL-ONES)]}
  (->IPv4Address 4 ip))


(defn- make-ipv4address-from-str
  [ip]
  {:pre [(spec/valid? ::isp/ipv4-address-spec ip)]}
  (-> (helpers/ipv4addr-str->int ip)
      make-ipv4address-from-int))

(defn make-ipv4address
  "Used to create and IPv4Address.
   'ip' can be an 'int' or a 'string'."
  [ip]
  (cond
    (int? ip)
    (make-ipv4address-from-int ip)
    (string? ip)
    (make-ipv4address-from-str ip)))



(defprotocol IPNetmask
  (to-prefixlen [this]
    "Converts an IPv[46]Netmask to a prefixlen (that is an int)")
  (to-hostmask [this]
    "Converts an IPv[46]Netmask to an IPv[46]Address representing the hostmask"))

(deftype IPv4Netmask
         [netmask* prefixlen*]
  IPAddrConv
  (to-int [_]
    (to-int netmask*))

  IPBase
  (to-string [_]
    (to-string netmask*))
  (toString [this]
    (to-string this))
  (version [_]
    (version netmask*))

  IPNetmask
  (to-prefixlen [_]
    prefixlen*)

  (to-hostmask [this]
    (-> (to-int this)
        Long/toBinaryString
        (str/replace "1" "")
        (str/replace "0" "1")
        (Long/parseUnsignedLong 2)
        ((fn [l] (->IPv4Address (version this) l))))))

(defmulti ^:private make-ipv4netmask
  "Creates an IPv4Netmask.
   Input can be an 'int' or a 'string'"
  class)
(defmethod ^:private make-ipv4netmask Long
  [netmask*]
  {:pre [(spec/valid? ::isp/ipv4-netmask-spec netmask*)]}
  (->IPv4Netmask (make-ipv4address netmask*) (helpers/netmaskv4-int->prefixlen netmask*)))
(defmethod ^:private make-ipv4netmask String
  [netmask*]
  (-> (helpers/ipv4addr-str->int netmask*)
      make-ipv4netmask))

(comment
  (let [netmask (make-ipv4netmask "255.255.254.0")]
    (class netmask))
  :rcf)

(defprotocol IPOutput
  (with-prefixlen [this]
    "Returns a string representation in the format: addr/prefixlen")
  (with-netmask [this]
    "Returns a string representation in the format: addr/netmask")
  (with-hostmask [this]
    "returns a string representation in the format: addr/hostmask"))


(defprotocol IPNetwork
  (base-address [this]
    "Returns the network address: IPv[46]Address type.")
  (netmask [this]
    "Returns the network netmask: IPv[46]Netmask type.")
  (broadcast-address [this]
    "Returns the network broadcast address: IPv[46]Address type.")
  (prefixlen [this]
    "Returns the network prefixlen: int")
  (hostmask [this]
    "Returns the network hostmask: IPv[46]Address")
  (hosts [this]
    "Returns the IPv[46]Address usable for host addressing in this network. Lazy seq.")
  (subnet-of? [this other]
    "Tests if the nerwork is a subnet of other network")
  (supernet-of? [this other]
    "Tests if the network is a superneet of other network.")
  (contains-ip? [this ip]
    "Tests if the network contains the IPv[46]Address")
  (overlaps? [this other]
    "Tests is the two IPv[46]Network overlap.")
  (subnets [this new-prefixlen]
    "Returns a list of IPv[46]Network with new-prefixlen which are subnets of this.")
  (supernet [this new-prefixlen]
    "Returns the supernet with new-prefixlen that contains the network.")
  (num-addresses [this]
    "Compute the number of addresses in the network.")
  (address-exclude [this other]
    "Returns a list of subnets of the network, with the exclusion of other")
  (to-seq [this]
    "Returns a lazy seq of all IPv[46]Address contained in the network."))

(defn- list-all-addresses* [net]
  (let [prefixlen (prefixlen net)
        broadcast-address (broadcast-address net)]
    (cond
      (and
       (= 32 prefixlen)
       (= 4 (version net)))
      [(base-address net)]
      (and
       (= 128 prefixlen)
       (= 6 (version net)))
      [(base-address net)]
      (and
       (= 31 prefixlen)
       (= 4 (version net)))
      [(base-address net) (next-one (base-address net))]
      (and
       (= 6 (version net))
       (=  127 prefixlen))
      [(base-address net) (next-one (base-address net))]

      :else
      (->> (iterate next-one (base-address net))
           (take-while #(let [d (compare % broadcast-address)]
                          (or
                           (neg? d)
                           (zero? d))))))))


(defn- broadcast-address-fn* [net]
  (cond
    (and (= 32 (prefixlen net))
         (= 4 (version net)))
    (base-address net)
    (and (= 128 (prefixlen net))
         (= 6 (version net)))
    (base-address net)
    :else
    (let [version* (version net)
          network-int (to-int (base-address net))
          hostmask-int (-> (hostmask net)
                           to-int)
          b (+ network-int hostmask-int)]
      (if (= 4 version*)
        (make-ipv4address b)
        (make-ipv6address [b (.scopeid (base-address net))])))))

(defn- subnet-of?* [this other]
  {:pre [= (class this) (class other)]}
  (let [base-compare (compare (base-address this) (base-address other))
        broadcast-compare (compare (broadcast-address this) (broadcast-address other))]
    (and
     (or
      (pos? base-compare)
      (zero? base-compare))
     (or
      (neg? broadcast-compare)
      (zero? broadcast-compare)))))

(defn- contains-ip?* [this ip]
  (let [base-compare (compare (base-address this) ip)
        broadcast-compare (compare  (broadcast-address this) ip)]
    (and
     (or
      (neg? base-compare)
      (zero? base-compare))
     (or
      (pos? broadcast-compare)
      (zero? broadcast-compare)))))

(defn- overlaps?* [this other]
  (let [[f s] (sort-by identity #(compare %1 %2) [this other])]
    (->> (compare  (base-address  s) (broadcast-address f))
         ((some-fn neg? zero?)))))

(defn- num-addresses* [this]
  (let [base-address-int  (to-int (base-address this))
        broadcast-address-int (to-int (broadcast-address this))]
    (-> (- broadcast-address-int base-address-int)
        inc)))

(defn- address-exclude* [this other]
  (when (not (supernet-of? this other))
    (throw (Exception. (format "%s not contained in %s" (with-prefixlen other) (with-prefixlen this)))))
  (if (zero? (compare this other))
    '()
    (let [new-prefixlen (inc (prefixlen this))
          [s1 s2] (subnets this new-prefixlen)]
      (cond
        (zero? (compare s1 other)) [s2]
        (zero? (compare s2 other)) [s1]
        (supernet-of? s1 other)
        (lazy-seq (cons s2  (address-exclude s1 other)))
        (supernet-of? s2 other)
        (lazy-seq (cons s1  (address-exclude s2 other)))   :else
        (throw (Exception. "Should not be there"))))))

(deftype IPv4Network [base-address* netmask*]
  IPScope
  (loopback? [this]
    (subnet-of? this @IPv4-LOOPBACK-NETWORK))
  (link-local? [this]
    (subnet-of? this @IPv4-LINK-LOCAL-NETWORK))
  (multicast? [this]
    (subnet-of? this @IPv4-MULTICAST-NETWORK))

  (private? [this]
    (if (some (partial subnet-of? this) @IPv4-PRIVATE-NETWORKS-EXCEPTIONS)
      false
      (-> (some (partial subnet-of? this) @IPv4-PRIVATE-NETWORKS)
          boolean)))

  (global? [this]
    (and (-> (private? this)
             not)
         (-> (subnet-of?  this @IPv4-PUBLIC-NETWORK)
             not)))
  (reserved? [this]
    (subnet-of? this @IPv4-RESERVED-NETWORK))
  (unspecified? [this]
    (and
     (-> (compare base-address* @IPv4-UNSPECIFIED-ADDRESS)
         zero?)
     (= 32 (prefixlen this))))

  IPNetwork
  (base-address [_]
    base-address*)
  (netmask [_]
    netmask*)
  (broadcast-address [this]
    (if (= 32 (prefixlen this))
      base-address*
      (let [network-int (to-int base-address*)
            hostmask-int (-> (hostmask this)
                             to-int)]
        (-> (+ network-int hostmask-int)
            make-ipv4address))))
  (prefixlen [_]
    (to-prefixlen netmask*))
  (hostmask [_]
    (to-hostmask netmask*))
  (hosts [this]
    (let [pf (prefixlen this)]
      (case  pf
        (32 31) (list-all-addresses* this)
        (-> (list-all-addresses* this)
            next
            butlast))))
  (to-seq [this]
    (list-all-addresses* this))

  (subnet-of? [this other]
    {:pre [(instance? IPv4Network other)]}
    (subnet-of?* this other))

  (supernet-of? [this other]
    {:pre (instance? IPv4Network other)}
    (subnet-of? other this))

  (contains-ip? [this ip]
    {:pre [(instance? IPv4Address ip)]}
    (contains-ip?* this ip))

  (overlaps? [this other]
    (overlaps?* this other))

  (subnets [this new-prefixlen]
    {:pre [(spec/valid? ::isp/ipv4-prefixlen-spec new-prefixlen) (<= (.prefixlen this) new-prefixlen)]}
    (let [new-netmask (-> (helpers/v4-prefixlen->int new-prefixlen)
                          make-ipv4netmask)
          b (broadcast-address this)
          new-network (IPv4Network.  base-address* new-netmask)]
      (->> (iterate next-one new-network)
           (take-while #(-> (compare  (broadcast-address %) b)
                            pos?
                            not)))))

  (supernet [this new-prefixlen]
    {:pre [(spec/valid? ::isp/ipv4-prefixlen-spec new-prefixlen) (<= new-prefixlen (.prefixlen this))]}
    (let [new-netmask (-> (helpers/v4-prefixlen->int new-prefixlen)
                          make-ipv4netmask)
          new-base-address (-> (bit-and (to-int base-address*) (to-int new-netmask))
                               make-ipv4address)
          new-network (IPv4Network.  new-base-address new-netmask)]
      (if (<=
           (-> (broadcast-address new-network) to-int)
           constants/IPv4-ALL-ONES)
        new-network
        nil)))
  (num-addresses [this]
    (num-addresses* this))
  (address-exclude [this other]
    {:pre [(instance? IPv4Network other)]}
    (address-exclude* this other))

  IPIter
  (next-one [this]
    (let [size (inc (- (to-int (broadcast-address this)) (to-int (base-address this))))
          new-start (+ size (-> (base-address this) (to-int)))
          new-broadcast (dec (+ size new-start))]
      (if (<= new-broadcast constants/IPv4-ALL-ONES)
        (IPv4Network.  (IPv4Address. (version base-address*) new-start) netmask*)
        nil)))
  IPBase
  (to-string [this]
    (with-prefixlen this))
  (toString [this]
    (to-string this))
  (version [_]
    (version base-address*))

  IPOutput
  (with-netmask [_]
    (str  base-address* "/" netmask*))
  (with-prefixlen [this]
    (str  base-address* "/" (prefixlen this)))
  (with-hostmask [this]
    (str  base-address* "/" (hostmask this)))

  Comparable
  (compareTo [this other]
    {:pre [(instance? IPv4Network other)]}
    (let [b1 base-address*
          b2 (base-address ^IPv4Network other)
          cmp (compare b1 b2)]
      (if (not (zero? cmp))
        cmp
        (compare (prefixlen this) (prefixlen other))))))


(defn make-ipv4network
  "Creates an IPv4Network.
   'arg' must be in form network/prefixlen or network/netmask"
  [arg]
  {:pre [(= 2 (count (str/split arg #"/")))
         (spec/valid? ::isp/ipv4-address-spec (nth (str/split arg #"/")  0))
         (or
          (spec/valid? ::isp/ipv4-prefixlen-spec (nth (str/split arg #"/")  1))
          (spec/valid? ::isp/ipv4-netmask-spec (nth (str/split arg #"/")  1)))]}

  (let [[network-str netmask-str] (str/split arg #"/")
        netmask-int (if (spec/valid? ::isp/ipv4-prefixlen-spec netmask-str)
                      (helpers/v4-prefixlen->int netmask-str)
                      (helpers/ipv4addr-str->int netmask-str))
        network-int (bit-and (helpers/ipv4addr-str->int network-str)  netmask-int)]
    (->IPv4Network  (make-ipv4address network-int) (make-ipv4netmask netmask-int))))

(comment
  (let [n (make-ipv4network "192.0.0.0/22")]
    (time  (->> (.subnets n 27)
                (map with-prefixlen))))
  (.num-addresses (make-ipv4network "1.2.3.0/24"))
  (->> (.address-exclude (make-ipv4network "1.2.3.0/24") (make-ipv4network "1.2.3.23/32"))
       (map #(.with-prefixlen %)))
  (->> (.subnets (make-ipv4network "1.2.3.24/31") 32)
       (map #(.with-prefixlen %)))
  :rcf)

(comment
  (-> (make-ipv4network "255.255.255.252/31")
      (.supernet 29)
      .with-prefixlen)

  (->> (make-ipv4network "10.47.1.0/29")
       .hosts
       (map #(str %)))

  :rcf)


(defprotocol IPInterface
  (network [this]
    "Returns the IPv[46]Network containing the IPv[46]Interface.")
  (address [this]
    "Returns the IPv[46]Address of the IPv[46]Interface."))

(deftype IPv4Interface [address* netmask*]
  IPScope
  (loopback? [this]
    (-> (network this)
        .loopback?))
  (link-local? [this]
    (-> (network this)
        .link-local?))
  (multicast? [this]
    (-> (network this)
        .multicast?))

  (private? [this]
    (-> (network this)
        .private?))
  (global? [_]
    (-> address*
        .global?))
  (reserved? [this]
    (-> (network this)
        .reserved?))
  (unspecified? [this]
    (-> (network this)
        .unspecified?))

  IPInterface
  (network [this]
    (make-ipv4network (.with-prefixlen this)))
  (address [_]
    address*)

  Comparable
  (compareTo [this other]
    {:pre [(instance? IPv4Interface other)]}
    (let [addr-cmp (compare address* (address other))
          net-cmp (compare (network this) (network other))]
      (if (zero? net-cmp)
        addr-cmp
        net-cmp)))

  IPIter
  (next-one [this]
    (let [ba (-> (network this) broadcast-address)]
      (if (zero? (compare address* ba))
        nil
        (IPv4Interface.  (next-one address*) netmask*))))

  IPBase
  (to-string [this]
    (with-prefixlen this))
  (toString [this]
    (to-string this))
  (version [_]
    (version address*))


  IPOutput
  (with-prefixlen [_]
    (-> (IPv4Network. address* netmask*)
        .with-prefixlen))
  (with-netmask [_]
    (-> (IPv4Network.  address* netmask*)
        .with-netmask))
  (with-hostmask [_]
    (-> (IPv4Network.  address* netmask*)
        .with-hostmask)))

(defn make-ipv4interface
  "Creates an IPv4Interface.
   'intf' must in format: ip/netmask or ip/prefixlen "
  [intf]
  {:pre  [(= 2 (count (str/split intf #"/")))
          (spec/valid? ::isp/ipv4-address-spec (nth (str/split intf #"/")  0))
          (or
           (spec/valid? ::isp/ipv4-prefixlen-spec (nth (str/split intf #"/")  1))
           (spec/valid? ::isp/ipv4-netmask-spec (nth (str/split intf #"/")  1)))]}
  (let [[ip-str netmask-str] (str/split intf #"/")
        netmask-int (if (spec/valid? ::isp/ipv4-prefixlen-spec netmask-str)
                      (helpers/v4-prefixlen->int netmask-str)
                      (helpers/ipv4addr-str->int netmask-str))
        ip-addr (make-ipv4address ip-str)
        netmask (make-ipv4netmask netmask-int)]
    (->IPv4Interface  ip-addr netmask)))

(extend-type nil
  IPIter
  (next-one [_]
    nil))


(comment
  (time
   (dotimes [_ 1000000] (make-ipv4network "1.1.1.1/24")))
  (-> (make-ipv4interface "192.0.2.7/29") next-one next-one)
  :rcf)

;;IPv6

(defprotocol IPv6Scope
  (site-local? [this]))

(defprotocol IPv6Specific
  (scopeid [this])
  (ipv4-mapped [this]
    "Returns the IPv4Address if it is an ipv4 mapped address. 
     Returns nil otherwise."))

(deftype IPv6Address [version* ip-int* scopeid*]
  IPv6Specific
  (scopeid [_]
    scopeid*)
  (ipv4-mapped [_]
    (let [b (biginteger ip-int*)]
      (if (== 0xffff (.shiftRight b 32))
        (-> (.and b (biginteger 0xffffffff))
            long
            make-ipv4address)
        nil)))


  IPBase
  (to-string [this]
    (let [scopestr (scopeid-str this)]
      (if-some [ipv4 (ipv4-mapped this)]
        (str "::ffff:" ipv4)
        (str (helpers/bigint->ipv6str ip-int*) scopestr))))
  (toString [this]
    (to-string this))
  (version [_]
    version*)

  IPScope
  (link-local? [this]
    (in? this @IPv6-LINK-LOCAL-NETWORK))
  (loopback? [_]
    (== ip-int* 1))
  (multicast? [this]
    (in? this @IPv6-MULTICAST-NETWORK))
  (private? [this]
    (if (some (partial in? this) @IPv6-PRIVATE-NETWORKS-EXCEPTIONS)
      false
      (-> (some (partial in? this) @IPv6-PRIVATE-NETWORKS)
          boolean)))
  (global? [this]
    (not (private? this)))
  (unspecified? [_]
    (== 0 ip-int*))
  (reserved? [this]
    (-> (some (partial in? this) @IPv6-RESERVED-NETWORKS)
        boolean))

  IPv6Scope
  (site-local? [this]
    (in? this @IPv6-SITE-LOCAL-NETWORK))

  IPAddrConv
  (to-int [_]
    ip-int*)

  IPIter
  (next-one [_]
    (if (<= @IPv6-ALL-ONES ip-int*)
      nil
      (IPv6Address. version* (inc ip-int*) scopeid*)))

  Comparable
  (compareTo [_ other]
    {:pre [(instance? IPv6Address other)]}
    (compare ip-int* (.ip-int* other)))

  IPAddress
  (in? [this net]
    (contains-ip? net this)))


(defmulti make-ipv6address class)
(defmethod make-ipv6address clojure.lang.BigInt
  [ip]
  (make-ipv6address (helpers/bigint->ipv6str ip)))
(defmethod make-ipv6address String
  [ip]
  (let [result (helpers/string->inet6address ip)]
    (if (instance? java.net.Inet6Address result)
      (->IPv6Address 6 (helpers/inet6address->bigint result) (.getScopeId result))
      (let [ipv4-int (-> (make-ipv4address (.getHostAddress result)) .to-int bigint)
            prefix-int (-> (bit-shift-left 0xffff 32) bigint)]
        (->IPv6Address 6 (+ ipv4-int prefix-int) 0)))))
(defmethod make-ipv6address clojure.lang.PersistentVector
  [[ip-int scope]]
  (->IPv6Address 6 ip-int scope))




(comment
  (-> (make-ipv6address "1080::8:800:200C:417A%123") to-string)
  (-> (make-ipv6address "1080::8:800:200C:417A%123") str)
  (-> (make-ipv6address "1080::8:800:200C:417A%0") scopeid)
  (-> (make-ipv6address "::FFFF:129.144.52.38")  str)
  (-> (make-ipv6address "Ffff:Ffff:Ffff:Ffff:Ffff:Ffff:Ffff:Fff8") str)
  :rcf)

(comment
  ((juxt
    site-local?
    loopback?
    link-local?
    multicast?) (make-ipv6address "1080::8:800:200C:417A"))
  @IPv6-ALL-ONES
  :rcf)

(deftype IPv6Netmask
         [netmask prefixlen*]
  IPAddrConv
  (to-int [_]
    (to-int netmask))

  IPBase
  (to-string [_]
    (to-string netmask))
  (toString [this]
    (to-string this))
  (version [_]
    (version netmask))

  IPNetmask
  (to-prefixlen [_]
    prefixlen*)

  (to-hostmask [_]
    (-> netmask
        to-int
        biginteger
        (.toString 2)
        (str/replace "1" "")
        (str/replace "0" "1")
        helpers/binstr->bigint
        ((fn [l] (IPv6Address. 6 l 0))))))

(defmulti ^:private make-ipv6netmask
  "Create an IPv6Netmask from a prefixlen (if arg is a Long).
   If arg is clojure.lang.BigInt: treats it as an int representation of the netmask.
   "
  class)
(defmethod ^:private make-ipv6netmask clojure.lang.BigInt
  [mask]
  (->IPv6Netmask  (-> mask
                      helpers/bigint->ipv6str
                      make-ipv6address)   (helpers/bigint->prefixlen mask)))
(defmethod ^:private make-ipv6netmask Long
  [pl]
  {:pre [(spec/valid? ::isp/ipv6-prefixlen-spec pl)]}
  (make-ipv6netmask (helpers/v6-prefixlen->bigint pl)))
(defmethod ^:private make-ipv6netmask java.math.BigInteger
  [mask]
  (make-ipv6netmask (bigint mask)))

(comment
  (-> (helpers/v6-prefixlen->bigint 125)
      helpers/bigint->prefixlen)
  (-> (make-ipv6netmask (helpers/v6-prefixlen->bigint 125)))
  (-> (helpers/v6-prefixlen->bigint 48)
      helpers/bigint->ipv6str
      make-ipv6address)
  :rcf)

(deftype IPv6Network [base-address* mask*]
  IPBase
  (version [_]
    (version base-address*))
  (to-string [_]
    (str  base-address* "/" (to-prefixlen mask*)))
  (toString [this]
    (to-string this))
  IPNetwork
  (base-address [_]
    base-address*)
  (netmask [_]
    mask*)
  (broadcast-address [this]
    (broadcast-address-fn* this))
  (prefixlen [_]
    (to-prefixlen mask*))
  (hostmask [_]
    (to-hostmask mask*))
  (hosts [this]
    (case (prefixlen this)
      (127 128) (to-seq this)
      (-> (to-seq this)
          next)))
  (subnet-of? [this other]
    {:pre [(instance? IPv6Network other)]}
    (subnet-of?* this other))
  (supernet-of? [this other]
    (subnet-of? other this))
  (contains-ip? [this ip]
    (contains-ip?* this ip))

  (overlaps? [this other]
    (overlaps?* this other))

  (subnets [this new-prefixlen]
    {:pre [(spec/valid? ::isp/ipv6-prefixlen-spec new-prefixlen)]}
    (when (< new-prefixlen (prefixlen this))
      (throw (Exception. "the new-prefixlen cannot be less than current one.")))
    (let [new-netmask (-> (helpers/v6-prefixlen->bigint new-prefixlen)
                          make-ipv6netmask)
          b (broadcast-address this)
          new-network (IPv6Network.  base-address* new-netmask)]
      (->> (iterate next-one new-network)
           (take-while #(-> (compare  (base-address %) b)
                            pos?
                            not)))))
  (supernet [this new-prefixlen]
    {:pre [(spec/valid? ::isp/ipv6-prefixlen-spec new-prefixlen)]}
    (when (< (prefixlen this) new-prefixlen)
      (throw (Exception. "new-prefixlen cannot be greater than current one.")))
    (let [new-netmask (-> (helpers/v6-prefixlen->bigint new-prefixlen)
                          make-ipv6netmask)
          new-base-address (-> (.and (biginteger (to-int base-address*)) (biginteger (to-int new-netmask)))
                               bigint
                               make-ipv6address)
          new-network (IPv6Network.  new-base-address new-netmask)]
      (if (<=
           (-> (broadcast-address new-network) to-int)
           @IPv6-ALL-ONES)
        new-network
        nil)))


  (num-addresses [this]
    (num-addresses* this))
  (address-exclude [this other]
    (address-exclude* this other))
  (to-seq [this]
    (list-all-addresses* this))
  IPIter
  (next-one [this]
    (let [base-int (to-int ^IPv6Address base-address*)
          size (inc (- (to-int (broadcast-address ^IPv6Network this))  base-int))
          new-start (+ size base-int)
          new-broadcast (dec (+ size new-start))]
      (if (<= new-broadcast @IPv6-ALL-ONES)
        (IPv6Network.  (IPv6Address. (version ^IPv6Address base-address*) new-start  (scopeid ^IPv6Address base-address*)) mask*)
        nil)))
  Comparable
  (compareTo [this other]
    {:pre [(instance? IPv6Network other)]}
    (let [cmp-addr (compare (base-address this) (base-address other))
          cmp-prefixlen (compare (prefixlen this) (prefixlen other))]
      (if (not (zero? cmp-addr))
        cmp-addr
        cmp-prefixlen)))

  IPOutput
  (with-prefixlen [this]
    (to-string this))
  (with-netmask [_]
    (str  base-address* "/"  mask*))
  (with-hostmask [_]
    (str  base-address* "/"  (to-hostmask mask*)))

  IPScope
  (link-local? [this]
    (subnet-of? this @IPv6-LINK-LOCAL-NETWORK))
  (loopback? [this]
    (and (== 128 (prefixlen this)) (== 1 (to-int base-address*))))
  (multicast? [this]
    (subnet-of? this @IPv6-MULTICAST-NETWORK))
  (private? [this]
    (if (some (partial subnet-of? this) @IPv6-PRIVATE-NETWORKS-EXCEPTIONS)
      false
      (-> (some (partial subnet-of? this) @IPv6-PRIVATE-NETWORKS)
          boolean)))
  (global? [this]
    (not (private? this)))
  (unspecified? [this]
    (and (unspecified? base-address*)
         (= 128 (prefixlen this))))
  (reserved? [this]
    (-> (some (partial subnet-of? this) @IPv6-RESERVED-NETWORKS)
        boolean))

  IPv6Scope
  (site-local? [this]
    (subnet-of? this @IPv6-SITE-LOCAL-NETWORK)))


(defn make-ipv6network
  "Create and IPv6Network. 
   'net' must be a string in format addr/prefixlen"
  [net]
  (let [[addr pl] (str/split net #"/")
        pl (if pl (parse-long pl) 128)
        ip6 (make-ipv6address addr)
        netmask-int (helpers/v6-prefixlen->bigint pl)
        addr-in (->  ip6
                     to-int)
        scope-str (scopeid-str ip6)]
    (->IPv6Network (->  (.and (biginteger addr-in) (biginteger netmask-int))
                        bigint
                        helpers/bigint->ipv6str
                        (str scope-str)
                        make-ipv6address) (make-ipv6netmask netmask-int))))

(comment
  (let [net (make-ipv6network "2001:658:22a:cafe::/123")]
    (->> (subnets net 126)
         (map to-string)))
  (-> (make-ipv6network "2001::/23")
      (subnets 24)
      (as-> $ (map to-string $)))

  (-> (make-ipv6network "2001::/64") with-prefixlen)

  (let [net1 (make-ipv6network "2001::/23")
        net2 (make-ipv6network "2001:3::/32")]
    (->> (address-exclude net1 net2)
         sort
         (map to-string)))

  :rcf)

(deftype IPv6Interface [address* netmask*]
  IPScope
  (loopback? [this]
    (-> (network this)
        .loopback?))
  (link-local? [this]
    (-> (network this)
        .link-local?))
  (multicast? [this]
    (-> (network this)
        .multicast?))

  (private? [this]
    (-> (network this)
        .private?))
  (global? [_]
    (-> address*
        .global?))
  (reserved? [this]
    (-> (network this)
        .reserved?))
  (unspecified? [this]
    (-> (network this)
        .unspecified?))

  IPv6Scope
  (site-local? [this]
    (-> (network this)
        .site-local?))


  IPInterface
  (network [this]
    (make-ipv6network (to-string this)))
  (address [_]
    address*)

  Comparable
  (compareTo [this other]
    {:pre [(instance? IPv6Interface other)]}
    (let [addr-cmp (compare address* (address other))
          net-cmp (compare (network this) (network other))]
      (if (zero? net-cmp)
        addr-cmp
        net-cmp)))

  IPIter
  (next-one [this]
    (let [ba (-> (network this) broadcast-address)]
      (if (zero? (compare address* ba))
        nil
        (IPv6Interface.  (next-one address*) netmask*))))

  IPBase
  (to-string [this]
    (with-prefixlen this))
  (toString [this]
    (to-string this))
  (version [_]
    (version address*))


  IPOutput
  (with-prefixlen [_]
    (-> (IPv6Network. address* netmask*)
        .with-prefixlen))
  (with-netmask [_]
    (-> (IPv6Network.  address* netmask*)
        .with-netmask))
  (with-hostmask [_]
    (-> (IPv6Network.  address* netmask*)
        .with-hostmask)))

(defn make-ipv6interface
  "Create and IPv6Interface. 
   'net' must be a string in format addr/prefixlen"
  [net]
  (let [[addr pl] (str/split net #"/")
        pl (if pl (parse-long pl) 128)
        addr (make-ipv6address addr)
        netmask (make-ipv6netmask pl)]
    (->IPv6Interface addr netmask)))

(comment
  (let [intf1 (make-ipv6interface "2001:658:22a:cafe:200:0:0:1/64")
        intf2 (make-ipv6interface "2001:658:22a:cafe:200:0:0:1")]
    (to-string (network intf1)))
  :rcf)
