(ns com.dmo-t.ipaddress.specs
  (:require
   [clojure.spec.alpha :as spec]
   [clojure.string :as str]
   [com.dmo-t.ipaddress.impl.constants :as constants]))

(defn- octet? [i]
  (cond
    (not (int? i)) false
    (neg? i) false
    (< 255 i) false
    :else
    true))

(defn- octet-str? [s]
  (cond
    (not (string? s)) false
    (not (parse-long s)) false
    :else
    (octet? (parse-long s))))


(defmulti ^{:private true} ipv4address? class)
(defmethod ^{:private true} ipv4address? Long
  [ip]
  (and
   (<= 0 ip)
   (<= ip constants/IPv4-ALL-ONES)))
(defmethod ^{:private true} ipv4address? String
  [ip]
  (cond
    (not (= 4 (count (str/split ip #"\.")))) false
    :else
    (every? octet-str? (str/split ip #"\."))))


(defmulti ^{:private true} v4-prefix-len? class)
(defmethod ^{:private true} v4-prefix-len? Long
  [l]
  (if (not (int? l))
    false
    (and (<= 0 l) (<= l 32))))
(defmethod ^{:private true} v4-prefix-len? String
  [l]
  (if-let [v (parse-long l)]
    (v4-prefix-len? v)
    false))

(defn- ^{:private true} binary-netmask?
  [b]
  (->  (or
        (re-matches #"^0+$" b)
        (re-matches #"^1+0*$" b))
       boolean))

(defmulti ^{:private true} v4-netmask? class)
(defmethod ^{:private true} v4-netmask? Long
  [ip]
  {:pre [(spec/valid? ::ipv4-address-spec ip)]}
  (-> (Long/toBinaryString ip)
      binary-netmask?))

(defmethod ^{:private true} v4-netmask? String
  [ip]
  {:pre [(spec/valid? ::ipv4-address-spec ip)]}
  (-> (reduce (fn [acc o]
                (str acc (-> (parse-long o) (Long/toBinaryString))))  "" (str/split ip #"\."))
      binary-netmask?))

(defmulti ^{:private true} v6-prefix-len? class)
(defmethod ^{:private true} v6-prefix-len? Long
  [l]
  (if (not (int? l))
    false
    (and (<= 0 l) (<= l 128))))
(defmethod ^{:private true} v6-prefix-len? String
  [l]
  (if-let [v (parse-long l)]
    (v4-prefix-len? v)
    false))


(spec/def ::ipv4-address-spec  ipv4address?)
(spec/def ::ipv4-prefixlen-spec v4-prefix-len?)
(spec/def ::ipv4-netmask-spec v4-netmask?)

(spec/def ::ipv6-prefixlen-spec v6-prefix-len?)

(comment
  (spec/valid? ::ipv4-prefixlen-spec 0)
  :rcf)




