(ns com.dmo-t.ipaddress.core
  (:require
   [clojure.spec.alpha :as spec]
   [clojure.string :as str]
   [com.dmo-t.ipaddress.impl.constants :as constants]
   [com.dmo-t.ipaddress.impl.helpers :as helpers]
   [com.dmo-t.ipaddress.specs :as isp]))

(set! *warn-on-reflection* false)

(declare make-ipv4address make-ipv4network)
(def IPv4-PRIVATE-NETWORKS (delay (mapv (fn [v] (make-ipv4network v)) constants/IPV4-PRIVATE-NETWORKS)))

(def IPv4-PRIVATE-NETWORKS-EXCEPTIONS (delay (mapv (fn [v] (make-ipv4network v)) constants/PRIVATE-NETWORKS-EXCEPTIONS)))
(def RESERVED-NETWORK (delay (make-ipv4network constants/RESERVED-NETWORK)))
(def UNSPECIFIED-ADDRESS (delay (make-ipv4address constants/UNSPECIFIED-ADDRESS)))
(def IPv4-LINK-LOCAL-NETWORK (delay (make-ipv4network constants/LINK-LOCAL-NETWORK))) (def IPv4-LOOPBACK-NETWORK (delay (make-ipv4network constants/LOOPBACK-NETWORK)))
(def IPv4-MULTICAST-NETWORK (delay (make-ipv4network constants/MULTICAST-NETWORK)))
(def IPv4-PUBLIC-NETWORK (delay (make-ipv4network constants/PUBLIC-NETWORK)))

(defprotocol IPIter
  (next-one [this]))

(defprotocol IPBase
  (to-string [this])
  (to-int [this]))
(defprotocol IPAddress
  (in? [this net]))
(defprotocol IPScope
  (loopback? [this])
  (private? [this])
  (global? [this])
  (reserved? [this])
  (link-local? [this])
  (multicast? [this])
  (unspecified? [this]))

(deftype IPv4Address [ip]
  IPAddress
  (in? [this net]
    (.contains-ip?  net this))
  IPBase
  (to-string [_]
    (helpers/int->ipv4addr-str ip))
  (to-int [_]
    ip)

  IPIter
  (next-one [_]
    (if (= constants/IPv4-ALL-ONES ip)
      nil
      (IPv4Address. (inc ip))))

  Comparable
  (compareTo [_ other]
    {:pre [instance? IPv4Address other]}
    (cond
      (< ip (.ip other)) -1
      (<  (.ip other) ip) 1
      :else
      0))

  IPScope
  (loopback? [this]
    (in? this @IPv4-LOOPBACK-NETWORK))
  (private? [this]
    (if (some (partial in? this) @IPv4-PRIVATE-NETWORKS-EXCEPTIONS)
      false
      (-> (some (partial in? this) @IPv4-PRIVATE-NETWORKS)
          boolean)))
  (global? [this]
    (and (-> (private? this)
             not)
         (-> (.contains-ip? @IPv4-PUBLIC-NETWORK this))))
  (reserved? [this]
    (in? this @RESERVED-NETWORK))
  (link-local? [this]
    (in? this @IPv4-LINK-LOCAL-NETWORK))
  (multicast? [this]
    (in? this @IPv4-MULTICAST-NETWORK))
  (unspecified? [this]
    (-> (compare this @UNSPECIFIED-ADDRESS)
        zero?)))

(defn- make-ipv4address-from-int
  [ip]
  {:pre [(<= 0 ip) (<= ip constants/IPv4-ALL-ONES)]}
  (->IPv4Address ip))


(defn- make-ipv4address-from-str
  [ip]
  {:pre [(spec/valid? ::isp/ipv4-address-spec ip)]}
  (-> (helpers/ipv4addr-str->int ip)
      make-ipv4address-from-int))

(defn make-ipv4address [ip]
  (cond
    (int? ip)
    (make-ipv4address-from-int ip)
    (string? ip)
    (make-ipv4address-from-str ip)))

(defprotocol IPNetmask
  (to-prefixlen [this])
  (to-hostmask [this]))

(deftype IPv4Netmask
         [netmask*]
  IPBase
  (to-string [_]
    (helpers/int->ipv4addr-str netmask*))
  (to-int [_]
    netmask*)

  IPNetmask
  (to-prefixlen [_]
    (->> netmask*
         Long/toBinaryString
         (filter #(= \1 %))
         count))
  (to-hostmask [_]
    (-> netmask*
        Long/toBinaryString
        (str/replace "1" "")
        (str/replace "0" "1")
        (Long/parseUnsignedLong 2)
        ->IPv4Address)))

(defmulti make-ipv4netmask class)
(defmethod make-ipv4netmask Long
  [netmask*]
  {:pre [(spec/valid? ::isp/ipv4-netmask-spec netmask*)]}
  (->IPv4Netmask netmask*))
(defmethod make-ipv4netmask String
  [netmask*]
  {:pre [(spec/valid? ::isp/ipv4-netmask-spec netmask*)]}
  (-> (helpers/ipv4addr-str->int netmask*)
      make-ipv4netmask))

(comment
  (let [netmask (make-ipv4netmask "255.255.254.0")]
    (class netmask))
  :rcf)

(defprotocol IPOutput
  (with-prefixlen [this])
  (with-netmask [this])
  (with-hostmask [this]))






(defprotocol IPNetwork
  (base-address [this])
  (netmask [this])
  (broadcast-address [this])
  (prefixlen [this])
  (hostmask [this])
  (hosts [this])
  (subnet-of? [this other])
  (supernet-of? [this other])
  (contains-ip? [this ip])
  (overlaps? [this other])
  (subnets [this new-prefixlen])
  (supernet [this new-prefixlen])
  (num-addresses [this])
  (address-exclude [this other])
  (to-seq [this]))

(defn- list-all-addresses [net]
  (let [prefixlen (prefixlen net)
        broadcast-address (broadcast-address net)]
    (condp = prefixlen
      32 [(base-address net)]
      31 [(base-address net) (next-one ^IPv4Address (base-address net))]
      (->> (iterate next-one (base-address net))
           (take-while #(let [d (compare % broadcast-address)]
                          (or
                           (neg? d)
                           (zero? d))))))))


(deftype IPv4Network [base-address* netmask*]
  IPScope
  (loopback? [this]
    (subnet-of? this @IPv4-LOOPBACK-NETWORK))
  (private? [this]
    (if (some (partial subnet-of? this) @IPv4-PRIVATE-NETWORKS-EXCEPTIONS)
      false
      (-> (some (partial subnet-of? this) @IPv4-PRIVATE-NETWORKS)
          boolean)))
  (global? [this]
    (and (-> (private? this)
             not)
         (-> (subnet-of?  this @IPv4-PUBLIC-NETWORK)
             not)))
  (reserved? [this]
    (subnet-of? this @RESERVED-NETWORK))
  (link-local? [this]
    (subnet-of? this @IPv4-LINK-LOCAL-NETWORK))
  (multicast? [this]
    (subnet-of? this @IPv4-MULTICAST-NETWORK))
  (unspecified? [this]
    (and
     (-> (compare base-address* @UNSPECIFIED-ADDRESS)
         zero?)
     (= 32 (prefixlen this))))

  IPNetwork
  (base-address [_]
    base-address*)
  (netmask [_]
    netmask*)
  (broadcast-address [this]
    (if (= 32 (prefixlen this))
      base-address*
      (let [network-int (to-int base-address*)
            hostmask-int (-> (hostmask this)
                             to-int)]
        (-> (+ network-int hostmask-int)
            make-ipv4address))))
  (prefixlen [_]
    (to-prefixlen netmask*))
  (hostmask [_]
    (to-hostmask netmask*))
  (hosts [this]
    (let [pf (prefixlen this)]
      (case  pf
        (32 31) (list-all-addresses this)
        (-> (list-all-addresses this)
            next
            butlast))))
  (to-seq [this]
    (list-all-addresses this))

  (subnet-of? [this other]
    {:pre [(instance? IPv4Network other)]}
    (let [base-compare (compare base-address* (base-address other))
          broadcast-compare (compare (broadcast-address this) (broadcast-address other))]
      (and
       (or
        (pos? base-compare)
        (zero? base-compare))
       (or
        (neg? broadcast-compare)
        (zero? broadcast-compare)))))

  (supernet-of? [this other]
    {:pre (instance? IPv4Network other)}
    (subnet-of? other this))
  (contains-ip? [this ip]
    {:pre [(instance? IPv4Address ip)]}
    (let [base-compare (compare base-address* ip)
          broadcast-compare (compare  (broadcast-address this) ip)]
      (and
       (or
        (neg? base-compare)
        (zero? base-compare))
       (or
        (pos? broadcast-compare)
        (zero? broadcast-compare)))))

  (overlaps? [this other]
    (let [[f s] (sort-by identity #(compare %1 %2) [this other])]
      (->> (compare  (base-address  s) (broadcast-address f))
           ((some-fn neg? zero?)))))

  (subnets [this new-prefixlen]
    {:pre [(spec/valid? ::isp/ipv4-prefixlen-spec new-prefixlen) (<= (.prefixlen this) new-prefixlen)]}
    (let [new-netmask (-> (helpers/v4-prefixlen->int new-prefixlen)
                          IPv4Netmask.)
          b (broadcast-address this)
          new-network (IPv4Network. base-address* new-netmask)]
      (->> (iterate next-one new-network)
           (take-while #(-> (compare  (broadcast-address %) b)
                            pos?
                            not)))))

  (supernet [this new-prefixlen]
    {:pre [(spec/valid? ::isp/ipv4-prefixlen-spec new-prefixlen) (<= new-prefixlen (.prefixlen this))]}
    (let [new-netmask (-> (helpers/v4-prefixlen->int new-prefixlen)
                          IPv4Netmask.)
          new-base-address (-> (bit-and (to-int base-address*) (to-int new-netmask))
                               IPv4Address.)
          new-network (IPv4Network.  new-base-address new-netmask)]
      (if (<=
           (-> (broadcast-address new-network) to-int)
           constants/IPv4-ALL-ONES)
        new-network
        nil)))
  (num-addresses [this]
    (let [base-address-int  (to-int base-address*)
          broadcast-address-int (to-int (broadcast-address this))]
      (-> (- broadcast-address-int base-address-int)
          inc)))
  (address-exclude [this other]
    {:pre [(instance? IPv4Network other)]}
    (when (not (supernet-of? this other))
      (throw (Exception. (format "%s not contained in %s" (with-prefixlen other) (with-prefixlen this)))))
    (if (zero? (compare this other))
      '()
      (let [new-prefixlen (inc (prefixlen this))
            [s1 s2] (subnets this new-prefixlen)]
        (cond
          (zero? (compare s1 other)) [s2]
          (zero? (compare s2 other)) [s1]
          (supernet-of? s1 other)
          (lazy-seq (cons s2  (address-exclude s1 other)))
          (supernet-of? s2 other)
          (lazy-seq (cons s1  (address-exclude s2 other)))   :else
          (throw (Exception. "Should not be there"))))))

  IPIter
  (next-one [this]
    (let [size (inc (- (to-int (broadcast-address this)) (to-int (base-address this))))
          new-start (+ size (-> (base-address this) (to-int)))
          new-broadcast (dec (+ size new-start))]
      (if (<= new-broadcast constants/IPv4-ALL-ONES)
        (IPv4Network. (IPv4Address. new-start) netmask*)
        nil)))
  IPOutput
  (with-netmask [_]
    (str (to-string base-address*) "/" (to-string netmask*)))
  (with-prefixlen [this]
    (str (to-string base-address*) "/" (prefixlen this)))
  (with-hostmask [this]
    (str (to-string base-address*) "/" (-> (hostmask this) to-string)))

  Comparable
  (compareTo [this other]
    {:pre [(instance? IPv4Network other)]}
    (let [b1 base-address*
          b2 (base-address ^IPv4Network other)
          cmp (compare b1 b2)]
      (if (not (zero? cmp))
        cmp
        (compare (prefixlen this) (prefixlen other))))))


(defn make-ipv4network
  "Entry must be in form network/prefixlen or network/netmask"
  [network-input]
  {:pre [(= 2 (count (str/split network-input #"/")))
         (spec/valid? ::isp/ipv4-address-spec (nth (str/split network-input #"/")  0))
         (or
          (spec/valid? ::isp/ipv4-prefixlen-spec (nth (str/split network-input #"/")  1))
          (spec/valid? ::isp/ipv4-netmask-spec (nth (str/split network-input #"/")  1)))]}

  (let [[network-str netmask-str] (str/split network-input #"/")
        netmask-int (if (spec/valid? ::isp/ipv4-prefixlen-spec netmask-str)
                      (helpers/v4-prefixlen->int netmask-str)
                      (helpers/ipv4addr-str->int netmask-str))
        network-int (bit-and (helpers/ipv4addr-str->int network-str)  netmask-int)]
    (->IPv4Network (make-ipv4address network-int) (make-ipv4netmask netmask-int))))

(comment
  (let [n (make-ipv4network "192.0.0.0/22")]
    (time  (->> (.subnets n 27)
                (map with-prefixlen))))
  (.num-addresses (make-ipv4network "1.2.3.0/24"))
  (->> (.address-exclude (make-ipv4network "1.2.3.0/24") (make-ipv4network "1.2.3.23/32"))
       (map #(.with-prefixlen %)))
  (->> (.subnets (make-ipv4network "1.2.3.24/31") 32)
       (map #(.with-prefixlen %)))
  :rcf)

(comment
  (-> (make-ipv4network "255.255.255.252/31")
      (.supernet 29)
      .with-prefixlen)

  (->> (make-ipv4network "10.47.1.0/29")
       .hosts
       (map #(to-string %)))

  :rcf)


(defprotocol IPInterface
  (network [this])
  (address [this]))

(deftype IPv4Interface [address* netmask*]
  IPScope
  (loopback? [this]
    (-> (network this)
        .loopback?))
  (private? [this]
    (-> (network this)
        .private?))
  (global? [_]
    (-> address*
        .global?))
  (reserved? [this]
    (-> (network this)
        .reserved?))
  (link-local? [this]
    (-> (network this)
        .link-local?))
  (multicast? [this]
    (-> (network this)
        .multicast?))
  (unspecified? [this]
    (-> (network this)
        .unspecified?))

  IPInterface
  (network [this]
    (make-ipv4network (.with-prefixlen this)))
  (address [_]
    address*)

  Comparable
  (compareTo [this other]
    {:pre [(instance? IPv4Interface other)]}
    (let [addr-cmp (compare address* (address other))
          net-cmp (compare (network this) (network other))]
      (if (zero? net-cmp)
        addr-cmp
        net-cmp)))

  IPIter
  (next-one [this]
    (let [ba (-> (network this) broadcast-address)]
      (if (zero? (compare address* ba))
        nil
        (IPv4Interface. (next-one address*) netmask*))))

  IPOutput
  (with-prefixlen [_]
    (-> (IPv4Network. address* netmask*)
        .with-prefixlen))
  (with-netmask [_]
    (-> (IPv4Network. address* netmask*)
        .with-netmask))
  (with-hostmask [_]
    (-> (IPv4Network. address* netmask*)
        .with-hostmask)))

(defn make-ipv4interface
  "intf-input must in format: ip/netmask or ip/prefixlen "
  [intf-input]
  {:pre  [(= 2 (count (str/split intf-input #"/")))
          (spec/valid? ::isp/ipv4-address-spec (nth (str/split intf-input #"/")  0))
          (or
           (spec/valid? ::isp/ipv4-prefixlen-spec (nth (str/split intf-input #"/")  1))
           (spec/valid? ::isp/ipv4-netmask-spec (nth (str/split intf-input #"/")  1)))]}
  (let [[ip-str netmask-str] (str/split intf-input #"/")
        netmask-int (if (spec/valid? ::isp/ipv4-prefixlen-spec netmask-str)
                      (helpers/v4-prefixlen->int netmask-str)
                      (helpers/ipv4addr-str->int netmask-str))
        ip-addr (make-ipv4address ip-str)
        netmask (make-ipv4netmask netmask-int)]
    (->IPv4Interface ip-addr netmask)))

(extend-type nil
  IPIter
  (next-one [_]
    nil))


(comment
  (time
   (dotimes [_ 1000000] (make-ipv4network "1.1.1.1/24")))
  (-> (make-ipv4interface "192.0.2.7/29") next-one next-one)
  :rcf)


