(ns com.dirtybrussels.clojure-plupload.core
  (:use [clojure.data json]
        [clojure.java io])
  (:import [javax.imageio ImageIO]
           [java.io InputStream OutputStream FileOutputStream
            ByteArrayInputStream ByteArrayOutputStream
            StringWriter PrintWriter]
           [java.util UUID]))


(defn plupload-css-js
  "Creates the hiccup \"markup\" vector to include the necessary javascript and
  css files in the html.
  google-jquery?: whether or not to include the google ajax api copy of jquery
                  1.6.4.min.js"
  [google-jquery?]
  (list
    [:link {:href "/plugins/plupload/js/jquery.plupload.queue/css/jquery.plupload.queue.css"
            :rel "stylesheet"
            :type "text/css"}]
    (if google-jquery?
      (list [:script {:src "https://ajax.googleapis.com/ajax/libs/jquery/1.6.4/jquery.min.js"
                      :type "text/javascript"}]
            [:script {:src "https://ajax.googleapis.com/ajax/libs/jqueryui/1.8.16/jquery-ui.min.js"
                      :type "text/javascript"}]))
    [:script {:src "/plugins/plupload/js/plupload.full.js"
              :type "text/javascript"}]
    [:script {:src "/plugins/plupload/js/jquery.plupload.queue/jquery.plupload.queue.js"
              :type "text/javascript"}]))


(defn plupload-response
  "Saves uploaded file to \"save-to-path\" location, and generates the
  appropriate HTTP response to a \"Plupload\" request map."
  ([request save-to-path filename]
   (let [params      (:params request)
         upload-path (str save-to-path "/" filename)
         multipart   (:multipart-params request)
         multipart?  (> (count multipart) 0)
         chunk-index (Integer/parseInt
                       (get (if multipart? multipart params) "chunk"))
         body        (if multipart?
                       (:tempfile (get multipart "file"))
                       (:body request))]
     (copy (try (cast InputStream body) (catch ClassCastException _ body))
           (cast OutputStream
                 (FileOutputStream. upload-path  (not= 0 chunk-index))))
     {:status  200
      :headers {"Content-Type" "application/json"}
      :body    (json-str {:status "success"})}))
  ([request save-to-path]
   (plupload-response request save-to-path (.toLowerCase (get (:params request) "name")))))
