(ns dentmetria.universe.core
  (:gen-class)
  (:refer-clojure :exclude [load])
  (:require
   [clojure.pprint :as pp]
   [honey.sql :as sql]
   [honey.sql.helpers :as h]
   [next.jdbc :as jdbc]
   [next.jdbc.result-set :as rs]
   [next.jdbc.connection :as connection]
   [taoensso.nippy :as nippy])
  (:import
   [clojure.lang IAtom IDeref IMeta IObj IRef IReference]
   [com.impossibl.postgres.api.jdbc PGConnection PGNotificationListener]
   [java.lang.ref WeakReference]
   [java.util.concurrent ConcurrentHashMap]
   [java.util.logging Level Logger]
   [org.apache.commons.io IOUtils]))

(def ^:private logger (Logger/getLogger "com.dentmetria"))
(.setLevel logger Level/INFO)

(defmulti write-obj (fn [m _] m))

(defmethod write-obj :nippy [_ obj]
  (nippy/freeze obj))

(defmulti read-obj (fn [m _] m))

(defmethod read-obj :nippy [_ stream]
  (let [obj (-> stream
                (IOUtils/toByteArray)
                (nippy/thaw))]
    (.close stream)
    obj))

(def ^{:dynamic true} multiverse (atom {}))

(defn- notify-universe [[client-id key]]
  (try
    (let [universes (get @multiverse key)]
      (when universes
        (doseq [u universes]
          (let [u (.get u)]
            (when (and u (not= client-id (.clientID u)))
              (.refresh u))))))
    true
    (catch Exception e
      (.error logger (str "Error notifying universe:"
                          client-id " / " key ": " e))
      false)))

(defn- prune-multiverse []
  (swap! multiverse
         (fn [m]
           (into {} (filter (comp not empty? second)
                            (map (fn [[k v]]
                                   [k (filter #(some? (.get %)) v)]) m))))))

(defn start-listener! [data-source]
  (let [get-connection (fn [data-source]
                         (-> (.getConnection data-source)
                             (.unwrap PGConnection)))
        listener (fn []
                   (reify PGNotificationListener
                     (notification [_this _processId _channelName payload]
                       (future
                         (prune-multiverse)
                         (notify-universe
                          (read-string payload))))))]
    (future
      (try
        (loop [conn (get-connection data-source)]
          (.addNotificationListener conn (listener))
          (doto (.createStatement conn)
            (.execute "LISTEN universe_changed")
            (.close))
          (doseq [[_key universes] @multiverse]
            (doseq [u universes]
              (let [u (.get u)]
                (when u
                  (.refresh u)))))
          (.info logger "Listener started.")
          (while (not (.isClosed conn))
            (Thread/sleep 1000))
          (.info logger "Listener connection was closed.")
          (recur (get-connection data-source)))
        (catch Exception e
          (.error logger (str "Error starting listener:" e)))))))

(defn- select-for-update [obj tx]
  (let [{:keys [key table-name rw-method]} obj
        q (-> {:select [:value]
               :from   [table-name]
               :where  [:= :key (write-obj rw-method key)]
               :for [:update]}
              (sql/format))
        rs (jdbc/execute!
            tx q {:builder-fn rs/as-unqualified-lower-maps})]
   (when (empty? rs)
     (throw (IllegalStateException.
             "Error reading universe. Perhaps it is deleted.")))
    (->> rs
         (first)
         (:value)
         (read-obj rw-method))))

(defn- insert-or-update [obj value tx]
  (let [{:keys [key table-name rw-method client-id]} obj
        query (-> (h/insert-into table-name)
                  (h/columns :key :value :updated_at)
                  (h/values [[(write-obj rw-method key)
                              (write-obj rw-method value)
                              (java.sql.Timestamp.
                               (System/currentTimeMillis))]])
                  (h/on-conflict :key)
                  (h/do-update-set {:value (write-obj rw-method value)
                                    :updated_at (java.sql.Timestamp.
                                                 (System/currentTimeMillis))})
                  (h/returning :value)
                  (sql/format))
        result (-> (jdbc/execute! tx query)
                   (first))]
    (jdbc/execute! tx [(str "NOTIFY universe_changed,'" (pr-str [client-id key]) "'")])
    result))

(defn- insert [k v conn table-name rw-method]
  (let [query (-> (h/insert-into table-name)
                  (h/columns :key :value)
                  (h/values [[(write-obj rw-method k)
                              (write-obj rw-method v)]])
                  (h/on-conflict :key)
                  (h/do-nothing)
                  (sql/format))]
    (jdbc/execute! conn query)))

;; next.jdbc with-transaction has problems with multithreading.
(defmacro with-transaction [[tx conn] & body]
  `(with-open [~tx (jdbc/get-connection ~conn)]
     (try
       (.setAutoCommit ~tx false)
       (.setTransactionIsolation ~tx java.sql.Connection/TRANSACTION_SERIALIZABLE)
       (try
         (let [result# (do ~@body)]
           (.commit ~tx)
           result#)
         (catch Exception e#
           (.rollback ~tx)
           (throw e#)))
       (catch Exception e#
         (throw e#)))))

(defn- notify-watchers [this old new]
  (doseq [[k v] (.getWatches this)]
    (v k this old new)))

(defn- update-key [this obj f & args]
  (let [{:keys [conn value]} obj
        [old-val new-val]
        (with-transaction [tx conn]
          (let [old-val (select-for-update obj tx)
                new-val (apply f old-val args)
                validator (.getValidator this)]
            (when (and validator
                       (not (validator new-val)))
              (throw (IllegalStateException. "Invalid reference state")))
            (insert-or-update obj new-val tx)
            [old-val new-val]))]
    (reset! value new-val)
    (notify-watchers this old-val new-val)
    new-val))

(defn- set-key [this obj new-val]
  (let [{:keys [conn value]} obj
        [old-val new-val]
        (with-transaction [tx conn]
          (let [old-val (select-for-update obj tx)
                validator (.getValidator this)]
            (when (and validator
                       (not (validator new-val)))
              (throw (IllegalStateException. "Invalid reference state")))
            (insert-or-update obj new-val tx)
            [old-val new-val]))]
    (reset! value new-val)
    (notify-watchers this old-val new-val)
    new-val))

(defn- delete-key [obj]
  (let [{:keys [conn key table-name rw-method]} obj]
    (->> {:delete-from table-name
          :where [:= :key (write-obj rw-method key)]}
         (sql/format)
         (jdbc/execute! conn))))

#_:clj-kondo/ignore
(definterface IUniverse
  (refresh [])
  (clientID [])
  (delete []))

(deftype Universe [obj]
  IUniverse
  (clientID [_]
    (obj :client-id))
  (refresh [this]
    (let [{:keys [conn value]} obj
          old-value @value
          new-value (with-transaction [tx conn]
                      (select-for-update obj tx))]
      (reset! value new-value)
      (notify-watchers this old-value new-value)))
  (delete [_this]
    (delete-key obj))

  IAtom
  (swap [this f]
    (update-key this obj f))
  (swap [this f arg]
    (update-key this obj f arg))
  (swap [this f arg1 arg2]
    (update-key this obj f arg1 arg2))
  (swap [this f arg1 arg2 more]
    (apply (partial update-key this obj f) arg1 arg2 more))
  (compareAndSet [this _oldval newval]
    (set-key this obj newval))
  (reset [this newval]
    (set-key this obj newval))

  IRef
  (setValidator [_ new-validator]
    (reset! (obj :validator) new-validator))
  (getValidator [_]
    @(obj :validator))
  (addWatch [this watch-key watch-fn]
    (.put (obj :watchers) watch-key watch-fn)
    this)
  (removeWatch [this watch-key]
    (.remove (obj :watchers) watch-key)
    this)
  (getWatches [_] (into {} (obj :watchers)))

  IDeref
  (deref [_this]
    @(obj :value))

  IMeta
  (meta [_]
    @(obj :meta-map))

  IObj
  (withMeta [_ new-meta]
    (let [new-meta (atom new-meta)]
      (Universe. (assoc obj :meta-map new-meta))))

  IReference
  (resetMeta [_ new-meta]
    (reset! (obj :meta-map) new-meta))
  (alterMeta [_ f args]
    (let [new-meta (apply f @(obj :meta-map) args)]
      (reset! (obj :meta-map) new-meta)))

  Object
  (finalize [_]))

(defn make [& args]
  (let [{:keys [key conn table-name init rw-method]} args
        table-name (or table-name :universe.store)
        rw-method (or rw-method :nippy)]
    (insert key init conn table-name rw-method)
    (let [obj {:key key
               :conn conn
               :table-name table-name
               :rw-method rw-method
               :client-id (random-uuid)
               :value (atom ::uninitialized)
               :watchers (ConcurrentHashMap.)
               :meta-map (atom {})
               :validator (atom nil)}
          u (Universe. obj)]
      (.refresh u)
      (swap! multiverse #(update % key conj (WeakReference. u)))
      u)))

(defn rm [universe]
  (.delete universe))

(defn ls [conn table-name]
  (let [q (-> {:select [:key [:%length.value] :updated_at]
               :from   [table-name]}
              (sql/format))
        rs (jdbc/execute!
            conn q {:builder-fn rs/as-unqualified-lower-maps})]
    (->> rs
         (map (fn [x] (update x :key #(read-obj :nippy %))))
         (pp/print-table))))

(comment
  (def p (connection/->pool
          com.zaxxer.hikari.HikariDataSource
          {:jdbcUrl
           (connection/jdbc-url {:dbtype "pgsql"
                                 :dbname "dentmetria"})
           :username "dentmetria" :password "dentmetria"}))

  (def l (start-listener! p))

  (def x (make :key :ml-bus
               :init {:a :b :c :d}
               :conn p))

  (def y (make :key :ml-bus
               :init {:a :b :c :d}
               :conn p))

  (reset! x 0)
  (swap! x inc)
  (rm x)

  (ls p :universe.store)

  ;;
  )
