(ns dentmetria.universe.core
  (:gen-class)
  (:refer-clojure :exclude [load])
  (:require
   [clojure.pprint :as pp]
   [honey.sql :as sql]
   [honey.sql.helpers :as h]
   [next.jdbc :as jdbc]
   [next.jdbc.result-set :as rs]
   [next.jdbc.connection :as connection]
   [taoensso.nippy :as nippy])
  (:import
   [clojure.lang IAtom IDeref IMeta IObj IRef IReference]
   [com.impossibl.postgres.api.jdbc PGConnection PGNotificationListener]
   [java.io Closeable]
   [java.util.concurrent ConcurrentHashMap]
   [java.util.logging Level Logger]
   [org.apache.commons.io IOUtils]))

(def ^:private logger (Logger/getLogger "com.dentmetria"))
(.setLevel logger Level/INFO)

(def ^:dynamic *table-name* :universe.store)

(def ^:dynamic *rw-method* :nippy)

(defmulti write-obj (fn [_] *rw-method*))

(defmethod write-obj :nippy [obj]
  (nippy/freeze obj))

(defmulti read-obj (fn [_] *rw-method*))

(defmethod read-obj :nippy [stream]
  (let [obj (-> stream
                (IOUtils/toByteArray)
                (nippy/thaw))]
    (.close stream)
    obj))

(def ^:private multiverse (atom {}))

(defn- notify-universe [[client-id key]]
  (try
    (let [universes (get @multiverse key)]
      (when universes
        (doseq [u universes]
          (when (not= client-id (.clientID u))
            (.refresh u)))))
    true
    (catch Exception e
      (.error logger (str "Error notifying universe:"
                          client-id " / " key ": " e))
      false)))

(defn start-listener! [data-source]
  (let [get-connection (fn [data-source]
                         (-> (.getConnection data-source)
                             (.unwrap PGConnection)))
        listener (fn []
                   (reify PGNotificationListener
                     (notification [_this _processId _channelName payload]
                       (future
                         (notify-universe
                          (read-string payload))))))]
    (future
      (try
        (loop [conn (get-connection data-source)]
          (.addNotificationListener conn (listener))
          (doto (.createStatement conn)
            (.execute "LISTEN universe_changed")
            (.close))
          (doseq [[_key universes] @multiverse]
            (doseq [u universes]
              (.refresh u)))
          (.info logger "Listener started.")
          (while (not (.isClosed conn))
            (Thread/sleep 1000))
          (.info logger "Listener connection was closed.")
          (recur (get-connection data-source)))
        (catch Exception e
          (.error logger (str "Error starting listener:" e)))))))

(defn- select-for-update [k tx]
  (let [q (-> {:select [:value]
               :from   [*table-name*]
               :where  [:= :key (write-obj k)]
               :for [:update]}
              (sql/format))
        rs (jdbc/execute!
            tx q {:builder-fn rs/as-unqualified-lower-maps})]
    (->> rs
         (first)
         (:value)
         (read-obj))))

(defn- insert-or-update [client-id key value tx]
  (let [query (-> (h/insert-into *table-name*)
                  (h/columns :key :value :updated_at)
                  (h/values [[(write-obj key)
                              (write-obj value)
                              (java.sql.Timestamp.
                               (System/currentTimeMillis))]])
                  (h/on-conflict :key)
                  (h/do-update-set {:value (write-obj value)
                                    :updated_at (java.sql.Timestamp.
                                                 (System/currentTimeMillis))})
                  (h/returning :value)
                  (sql/format))
        result (-> (jdbc/execute! tx query)
                   (first))]
    (jdbc/execute! tx [(str "NOTIFY universe_changed,'" (pr-str [client-id key]) "'")])
    result))

(defn- insert [k v conn]
  (let [query (-> (h/insert-into *table-name*)
                  (h/columns :key :value)
                  (h/values [[(write-obj k)
                              (write-obj v)]])
                  (h/on-conflict :key)
                  (h/do-nothing)
                  (sql/format))]
    (jdbc/execute! conn query)))

;; next.jdbc with-transaction has problems with multithreading.
(defmacro with-transaction [[tx conn] & body]
  `(with-open [~tx (jdbc/get-connection ~conn)]
     (try
       (.setAutoCommit ~tx false)
       (.setTransactionIsolation ~tx java.sql.Connection/TRANSACTION_SERIALIZABLE)
       (try
         (let [result# (do ~@body)]
           (.commit ~tx)
           result#)
         (catch Exception e#
           (.rollback ~tx)
           (throw e#)))
       (catch Exception e#
         (throw e#)))))

(defn- notify-watchers [this old new]
  (doseq [[k v] (.getWatches this)]
    (v k this old new)))

(defn- update-key [this state f & args]
  (let [{:keys [conn key value client-id]} state
        [old-val new-val]
        (with-transaction [tx conn]
          (let [old-val (select-for-update key tx)
                new-val (apply f old-val args)
                validator (.getValidator this)]
            (when (and validator
                       (not (validator new-val)))
              (throw (IllegalStateException. "Invalid reference state")))
            (insert-or-update client-id key new-val tx)
            [old-val new-val]))]
    (reset! value new-val)
    (notify-watchers this old-val new-val)
    new-val))

(defn- set-key [this state new-val]
  (let [{:keys [conn key value client-id]} state
        [old-val new-val]
        (with-transaction [tx conn]
          (let [old-val (select-for-update key tx)
                validator (.getValidator this)]
            (when (and validator
                       (not (validator new-val)))
              (throw (IllegalStateException. "Invalid reference state")))
            (insert-or-update client-id key new-val tx)
            [old-val new-val]))]
    (reset! value new-val)
    (notify-watchers this old-val new-val)
    new-val))

^{:clj-kondo/ignore true}
(definterface IRefresh
  (refresh [])
  (clientID []))

(deftype Universe [state]
  IRefresh
  (clientID [_]
    (state :client-id))
  (refresh [this]
    (let [{:keys [conn key value]} state
          old-value @value
          new-value (with-transaction [tx conn]
                      (select-for-update key tx))]
      (reset! value new-value)
      (notify-watchers this old-value new-value)))

  IAtom
  (swap [this f]
    (update-key this state f))
  (swap [this f arg]
    (update-key this state f arg))
  (swap [this f arg1 arg2]
    (update-key this state f arg1 arg2))
  (swap [this f arg1 arg2 more]
    (apply (partial update-key this state f) arg1 arg2 more))
  (compareAndSet [this _oldval newval]
    (set-key this state newval))
  (reset [this newval]
    (set-key this state newval))

  IRef
  (setValidator [_ new-validator]
    (reset! (state :validator) new-validator))
  (getValidator [_]
    @(state :validator))
  (addWatch [this watch-key watch-fn]
    (.put (state :watchers) watch-key watch-fn)
    this)
  (removeWatch [this watch-key]
    (.remove (state :watchers) watch-key)
    this)
  (getWatches [_] (into {} (state :watchers)))

  IDeref
  (deref [_this]
    @(state :value))

  IMeta
  (meta [_]
    @(state :meta-map))

  IObj
  (withMeta [_ new-meta]
    (let [new-meta (atom new-meta)]
      (Universe. (assoc state :meta-map new-meta))))

  IReference
  (resetMeta [_ new-meta]
    (reset! (state :meta-map) new-meta))
  (alterMeta [_ f args]
    (let [new-meta (apply f @(state :meta-map) args)]
      (reset! (state :meta-map) new-meta)))

  Closeable
  (close [this]
    (swap! multiverse (fn [mv] (update mv :ml-bus (fn [xs] (remove #(= this %) xs))))))

  Object
  (finalize [_] ))

(defn make [& args]
  (let [{:keys [key conn init]} args]
    (insert key init conn)
    (let [value (with-transaction [tx conn]
                  (select-for-update key tx))
          state {:key key
                 :conn conn
                 :client-id (random-uuid)
                 :value (atom value)
                 :watchers (ConcurrentHashMap.)
                 :meta-map (atom {})
                 :validator (atom nil)}
          u (Universe. state)]
      (swap! multiverse #(update % key conj u))
      u)))

(defn ls [conn]
  (->> {:select [:key [:%length.value] :updated_at]
        :from   [*table-name*]}
       (sql/format)
       (jdbc/execute! conn)
       (map #(update % :store/key read-obj))
       (pp/print-table)))

(defn rm [conn key]
  (->> {:delete-from *table-name*
        :where [:= :key (write-obj key)]}
       (sql/format)
       (jdbc/execute! conn)))

(comment
  (def p (connection/->pool
          com.zaxxer.hikari.HikariDataSource
          {:jdbcUrl
           (connection/jdbc-url {:dbtype "pgsql"
                                 :dbname "dentmetria"})
           :username "dentmetria" :password "dentmetria"}))

  (def l (start-listener! p))

  (def x (make :key :ml-bus
               :init {:a :b :c :d}
               :conn p))

  (def y (make :key :ml-bus
               :init {:a :b :c :d}
               :conn p))

  (reset! x 0)
  (swap! x inc)

  (ls p)

  ;;
  )
