;; https://cljdoc.org/d/org.clojure/core.cache/1.1.234/doc/using-core-cache
(ns dentmetria.universe.cache
  (:gen-class)
  (:require [dentmetria.universe.core :as core])
  (:import [clojure.lang IDeref]))

^{:clj-kondo/ignore true}
(definterface ICache
  ;;This is the protocol describing the basic cache capability.

  ;; Retrieve the value associated with `e` if it exists, else `nil` in
  ;; the 2-arg case.  Retrieve the value associated with `e` if it exists,
  ;; else `not-found` in the 3-arg case.
  (lookup [e])
  (lookup [e not-found])
  ;; Checks if the cache contains a value associated with `e`
  (has   [e])
  ;; Is meant to be called if the cache is determined to contain a value
  ;; associated with `e`
  (hit    [e])
  ;; Is meant to be called if the cache is determined to **not** contain a
  ;; value associated with `e`
  (miss   [e ret])
  ;; Removes an entry from the cache
  (evict  [e]))

(defn- evict-by-ttl [cache ttl-ms]
  (let [evictions (->> cache
                       (filter
                        (fn [[_ [_ created-at]]]
                          (> (- (System/currentTimeMillis) created-at) ttl-ms)))
                       (map first))]
    (apply dissoc cache evictions)))

(deftype TTLCache [cache ttl-ms]
  ICache
  (lookup [this key]
    (let [val (.lookup this key ::missing)]
      (when-not (= ::missing val) val)))
  (lookup [this key not-found]
    (if (.has this key)
      (first (get @cache key))
      not-found))
  (has [_ key]
    (when-let [[_ created_at] (get @cache key)]
      (let [age (- (System/currentTimeMillis) created_at)]
        (<= age ttl-ms))))
  (hit [this key]
    (let [val (.lookup this key ::missing)]
      (when-not (= ::missing val)
        ;; reset ttl
        (.miss this key val)))
    this)
  (miss [this key value]
    (swap! cache (fn [cache]
                   (let [cache (evict-by-ttl cache ttl-ms)]
                     (assoc cache key [value (System/currentTimeMillis)]))))
    this)
  (evict [this key]
    (swap! cache dissoc key)
    this)
  IDeref
  (deref [_this]
    (update-vals @cache first)))

(defn ttl [& args]
  (let [args (-> (apply hash-map args)
                 (dissoc :init)
                 (assoc :init {}))
        {:keys [ttl-ms]} args]
    (TTLCache. (apply core/make (-> args seq flatten)) ttl-ms)))

(defn lookup
  ([c k]
   (.lookup c k))
  ([c k not-found]
   (.lookup c k not-found)))

(defn has? [c k]
  (.has c k))

(defn hit [c k]
  (.hit c k))

(defn miss [c k v]
  (.miss c k v))

(defn evict [c k]
  (.evict c k))

(comment
  (require '[next.jdbc.connection :as connection])

  (def p (connection/->pool
          com.zaxxer.hikari.HikariDataSource
          {:jdbcUrl
           (connection/jdbc-url {:dbtype "pgsql"
                                 :dbname "dentmetria"})
           :username "dentmetria" :password "dentmetria"}))


  (def c (ttl :conn p
                    :key :b
                    :ttl 1000))
  ;;
  )
