;;; Author: David Goldfarb (deg@degel.com)
;;; Copyright (c) 2017, David Goldfarb

(ns sodium.utils
  (:require
   [clojure.set]
   [clojure.spec.alpha :as s]
   [clojure.string :as str]))


(defn ci-compare
  "Case-insensitive string compare"
  [s1 s2]
  (compare (str/upper-case s1) (str/upper-case s2)))

(defn ci-sort
  "Case-insensitive string sort"
  [l]
  (sort-by str/upper-case l))

(defn ci-includes?
  "Case-insensitive string inclusion test"
  [s substr]
  (str/includes? (str/upper-case s) (str/upper-case substr)))


(defn validate
  "Like s/valid?, but show the error like s/assert. Useful for pre-conditions."
  [spec x]
  (or (s/valid? spec x)
      (s/explain spec x)))

(defn vconcat
  "Like concat, but return a vector."
  [& vecs]
  (vec (apply concat vecs)))

(s/def ::event-vector (s/cat :event keyword? :params (s/* any?)))

(defn unpredicate
  "Remove trailing '?' from predicate, to make suitable for JavaScript"
  [s]
  (if (str/ends-with? s "?")
    (subs s 0 (-> s count dec))
    s))

(defn camelize-str
  "Convert a string from ClojureScript to JavaScript conventions.
  - Replace hyphens with camelCase
  - Remove trailing '?'"
  [s]
  (let [[first-word & more] (str/split (unpredicate s) "-")]
    (if more
      (str first-word (str/join (map str/capitalize more)))
      first-word)))

(defn camelize-key
  "Convert a keyword from ClojureScript to JavaScript conventions.
  - Replace hyphens with camelCase
  - Remove trailing '?'
  - Preserve namespace"
  [k]
  (keyword (namespace k)
           (camelize-str (name k))))

(defn camelize-map-keys
  "Convert a map from ClojureScript to JavaScript conventions. Change the map
  keys, but leave the values alone.  For convenience, you can pass in a seq
  of keywords that must be excluded (left unchanged)."
  [m & {:keys [exclude]}]
  (reduce-kv (fn [m k v]
               (assoc m
                      (if (some #{k} exclude) k (camelize-key k))
                      v))
             {} m))

(defn err
  "Simple helper to show an error message in Clojure or ClojureScript"
  [& strings]
  (apply #?(:clj (partial println "Error: ") :cljs js/console.error) strings))


(defn all-keys-valid?
  "Check that each of keys is present in universe.
  If not, check more carefully to report a useful warning to the console.
  [TODO] Note that this is not a 100% logical place for this function. It is
  ClojureScript runtime support for code generated by the defcontrol macro."
  [keys universe]
  {:pre [(validate set? universe)]}
  (if (clojure.set/subset? (set keys) (set universe))
    true
    (run! (fn [key]
            (when-not (contains? universe key)
              (err "Keyword " key " is invalid. Not in " universe)))
          keys)))
