(ns ^{:doc "Collection of useful tools"}
    kafkian.utility-belt
  (:require
   [kafkian.consumer :refer [consumer messages seek subscribe]]
   [clojure.core.async :as async])
  )


(def default-consumer-conf {"bootstrap.servers" "localhost:9092"
                            "auto.offset.reset" "latest"
                            "enable.auto.commit" "false"
                            "key.deserializer" "org.apache.kafka.common.serialization.StringDeserializer"
                            "value.deserializer" "org.apache.kafka.common.serialization.StringDeserializer"})



(defn- topic-consumer [config {:keys [topic partitions offset] :as tps}]
  (let [conf (merge {"group.id" (str "kafkian-consumer-" topic "-" (rand-int 100000))}
                    config)
        c (consumer conf)
        tp-vec (mapv #(assoc {} :topic topic :partition %) partitions)]
    (subscribe c [tps])
    (if (some? offset)
      (seek c tp-vec offset))
    c))

(defn- msg-chan [conf {:keys [max-msgs] :as tps-m}]
  (let [ch (async/chan)]
    ;;NOTE Ideally the go block below should be async/thread as polling kafka is a
    ;;     BLOCKING I/O operation. Currently keeping it async/go
    ;;     as we are using the blocking/timeout default of 1s on polling
    ;;     kafka. Might change to async/thread in the future
    (async/go
      (with-open [c (topic-consumer conf tps-m)]
        (if (nil? max-msgs)
          ;;poll once
          (do
            (run! #(async/put! ch %) (messages c))
            (async/close! ch))
          ;;continously poll until max-msgs read or there's no more msgs
          (loop [msgs (take max-msgs (messages c))
                 msgs-taken (count msgs)
                 msgs-left (- max-msgs msgs-taken)]
            (run! #(async/put! ch %) msgs)
            (if (or  (= 0 msgs-left) (= 0 msgs-taken))
              (async/close! ch)
              (let [m (take msgs-left (messages c))
                    m-taken (count m)
                    m-left (- msgs-left m-taken)]
                (recur m m-taken m-left)))))))
    ch))



(defn browse
  ""
  [tps-maps & {:keys[config]}]
  (let [conf (merge default-consumer-conf config)
        all-chs (reduce #(conj %1 (msg-chan conf %2)) [] tps-maps)
        merged-ch (async/merge all-chs)]

    (let [all-messages (async/<!! (async/into [] merged-ch))
          update-fn (fn [m-v m] (conj (or m-v []) m))]
      (reduce #(update-in %1 [(:topic %2) (:partition %2)] update-fn %2) {} all-messages)
      ;; all-messages
      )))
