(ns coconut.alpha.summarizing
  (:require
    [clojure.core.match :as ccm]
    [coconut.alpha.aggregation :as aggregation]
    [coconut.alpha.running :as running]
    ))

(defn summarize
  ([state event]
   (ccm/match [[(::current-status (::current-test state) ::not-executing)
                (::running/type event)]]

     [[::not-executing
       ::running/suite-started]]
     #::aggregation{:events [#::{:type ::suite-started
                                 :total-number-of-tests (::running/total-number-of-tests event)}]
                    :state #::{:context []
                               :current-test nil
                               :number-of-passing-tests 0
                               :number-of-failing-tests 0
                               :suite-started-at (::running/current-time-millis event)}}

     [[::not-executing
       ::running/context-started]]
     #::aggregation{:events [#::{:type ::context-started
                                 :subject (::running/subject event)}]
                    :state (update state
                                   ::context
                                   conj
                                   (::running/subject event))}

     [[::not-executing
       ::running/test-marked-pending]]
     #::aggregation{:events [#::{:type ::test-pending
                                 :line-number (::running/definition-line-number event)
                                 :context (::context state)
                                 :description (::running/description event)
                                 :pending-reason (::running/pending-reason event)
                                 :duration-in-milliseconds 0}]
                    :state state}

     [[::not-executing
       ::running/test-started]]
     #::aggregation{:events []
                    :state (assoc state
                                  ::current-test #::{:current-status ::nothing-asserted
                                                     :relevant-assertion-failed-event nil
                                                     :namespace-name (::running/namespace-name event)
                                                     :line-number (::running/definition-line-number event)})}

     [[::nothing-asserted
       ::running/assertion-passed]]
     #::aggregation{:events []
                    :state (assoc-in state
                                     [::current-test ::current-status] ::passing)}

     [[(:or ::nothing-asserted
            ::passing)
       ::running/assertion-failed]]
     #::aggregation{:events []
                    :state (-> state
                               (assoc-in [::current-test ::current-status] ::failing)
                               (assoc-in [::current-test ::relevant-assertion-failed-event] event))}

     [[(:or ::nothing-asserted
            ::passing
            ::failing)
       ::running/test-timed-out]]
     #::aggregation{:events [#::{:type ::test-timed-out
                                 :context (::context state)
                                 :namespace-name (::namespace-name (::current-test state))
                                 :line-number (::line-number (::current-test state))
                                 :description (::running/description event)}]
                    :state (-> state
                               (assoc ::current-test nil)
                               (update ::number-of-failing-tests inc))}

     [[(:or ::nothing-asserted
            ::passing)
       ::running/test-threw-exception]]
     #::aggregation{:events [#::{:type ::test-threw-exception
                                 :namespace-name (::namespace-name (::current-test state))
                                 :line-number (::line-number (::current-test state))
                                 :context (::context state)
                                 :description (::running/description event)
                                 :exception (::running/exception event)}]
                    :state (-> state
                               (assoc ::current-test nil)
                               (update ::number-of-failing-tests inc))}

     [[::nothing-asserted
       ::running/test-finished]]
     #::aggregation{:events [#::{:type ::test-pending
                                 :line-number (::line-number (::current-test state))
                                 :context (::context state)
                                 :description (::running/description event)
                                 :pending-reason "nothing has been asserted"}]
                    :state (assoc state
                                  ::current-test nil)}

     [[::failing
       (:or ::running/test-finished
            ::running/test-threw-exception)]]
     #::aggregation{:events [#::{:type ::test-failed
                                 :namespace-name (::namespace-name (::current-test state))
                                 :line-number (or (:coconut.matchers/line-number (::relevant-assertion-failed-event (::current-test state)))
                                                  (::line-number (::current-test state)))
                                 :context (::context state)
                                 :description (::running/description event)
                                 :expected (::running/expected (::relevant-assertion-failed-event (::current-test state)))
                                 :actual (::running/actual (::relevant-assertion-failed-event (::current-test state)))}]
                    :state (-> state
                               (assoc ::current-test nil)
                               (update ::number-of-failing-tests inc))}

     [(:or [::passing
            ::running/assertion-passed]
           [::failing
            (:or ::running/assertion-passed
                 ::running/assertion-failed)])]
     #::aggregation{:events []
                    :state state}

     [[::passing
       ::running/test-finished]]
     #::aggregation{:events [#::{:type ::test-passed
                                 :namespace-name (::namespace-name (::current-test state))
                                 :line-number (::line-number (::current-test state))
                                 :context (::context state)
                                 :description (::running/description event)}]
                    :state (-> state
                               (assoc ::current-test nil)
                               (update ::number-of-passing-tests inc))}

     [[::not-executing
       ::running/context-finished]]
     #::aggregation{:events [#::{:type ::context-finished
                                 :subject (::running/subject event)}]
                    :state (update state ::context pop)}

     [[::not-executing
       ::running/suite-finished]]
     #::aggregation{:events [#::{:type ::suite-finished
                                 :success? (and (zero? (::number-of-failing-tests state))
                                                (not (zero? (::number-of-passing-tests state))))
                                 :warnings (::warnings state)
                                 :duration-in-milliseconds (- (::running/current-time-millis event)
                                                              (::suite-started-at state))}]
                    :state state}

     [[_ ::running/warning]]
     #::aggregation{:events []
                    :state (update state
                                   ::warnings conj #::{:type ::warning
                                                       :namespace-name (::running/namespace-name event)
                                                       :test-definition-line-number (::running/test-definition-line-number event)
                                                       :around-each-definition-line-number (::running/around-each-definition-line-number event)
                                                       :message (::running/message event)})})))
