(ns coconut.alpha.running
  #?(:cljs (:require-macros [cljs.core.async.macros :as async]))
  (:require
    [clojure.core.async :as async :refer [<! >!]]
    [coconut.alpha.core :as core]
    [coconut.alpha.matchers :as matchers]
    [coconut.alpha.platform :as platform]
    [coconut.alpha.query :as query]
    ))

(def ^{:dynamic true} *current-environment*
  {::default? true
   :coconut.alpha/assert-that
   (fn [& _]
     (throw (platform/exception
              (transduce (interpose \newline)
                         str
                         (vector "The assertion function is not bound. It's possible you're invoking the"
                                 "#'coconut.alpha/assert-that function within a test which is defined using"
                                 "#'coconut.alpha/it*. For those tests you need to invoke the assertion function"
                                 "which is injected directly into the test instead of the dynamically-bound"
                                 "version. This function is available at the :coconut.alpha/assert-that key"
                                 "in the environment map which is injected into every test.")))))

   :coconut.alpha/done
   (fn [& _]
     (throw (platform/exception
              (transduce (interpose \newline)
                         str
                         (vector "The done function is not bound. It's possible you're invoking the"
                                 "#'coconut.alpha/done function within a test which is defined using"
                                 "#'coconut.alpha/it*. For those tests you need to invoke the done function"
                                 "which is injected directly into the test instead of the dynamically-bound"
                                 "version. This function is available at the :coconut.alpha/done key"
                                 "in the environment map which is injected into every test.")))))})

(defn current-environment
  ([]
   (if-not (::default? *current-environment*)
     *current-environment*
     (throw (platform/exception
              (transduce (interpose \newline)
                         str
                         (vector "No current environment is bound. It's possible you're invoking the"
                                 "#'coconut.alpha/current-environment function within a test which is"
                                 "defined using #'coconut.alpha/it*. For those tests, the environment is"
                                 "the single argument that is passed to the test function.")))))))

(defn create-assertion-function
  ([assertion-event-atom]
   (fn assert-that
     ([actual matcher]
      (swap! assertion-event-atom
             conj
             (if-let [failure (matchers/evaluate-matcher matcher actual)]
               #::{:type ::assertion-failed
                   :expected (:expected failure)
                   :actual (:actual failure)
                   :current-time-millis (platform/current-time-millis)
                   :coconut.matchers/line-number (:coconut.matchers/line-number failure)}
               #::{:type ::assertion-passed
                   :current-time-millis (platform/current-time-millis)}))))))

(defn create-done-function
  ([done-channel]
   (fn done
     ([]
      (async/close! done-channel))
     ([e]
      (async/go
        (do (>! done-channel e)
            (async/close! done-channel)))))))

(defn with-pending-information
  ([execution-state component]
   (if (nil? (::core/pending? component))
     execution-state
     (assoc execution-state
            ::pending? (::core/pending? component)
            ::pending-reason (::core/pending-reason component)))))

(defn with-asynchronous-information
  ([execution-state component]
   (if (nil? (::core/asynchronous? component))
     execution-state
     (assoc execution-state
            ::asynchronous? (::core/asynchronous? component)
            ::timeout-in-milliseconds (::core/timeout-in-milliseconds component)))))

(defn with-composed-around-each-functions
  ([execution-state component]
   (update execution-state
           ::around-each
           #(transduce (map ::core/function)
                       (fn
                         ([_] _)
                         ([f g]
                          (fn [it]
                            (f (fn x
                                 ([]
                                  (x (fn [])))
                                 ([a1]
                                  (g (fn y
                                       ([]
                                        (y (fn [])))
                                       ([a2]
                                        (it (fn []
                                              (do (a2)
                                                  (a1)))))))))))))
                       %
                       (::core/components (::core/around-each component))))))

(defn with-composed-environment
  ([execution-state component]
   (update execution-state
           ::environment
           #(transduce (map ::core/function)
                       (fn
                         ([_] _)
                         ([f g]
                          (comp g f)))
                       %
                       (::core/components (::core/environment component))))))

(defn copy-to-channel
  ([channel collection]
   (let [close-channel? false]
     (async/onto-chan channel collection close-channel?))))

(defmulti run-component
  (fn [event-channel execution-state component]
    (::core/component-type component)))

(defmethod run-component
  ::query/result
  ([event-channel execution-state component]
   (async/go
     (<! (run-component event-channel
                        execution-state
                        (::core/sub-components component))))))

(defmethod run-component
  ::core/collection
  ([event-channel execution-state component]
   (async/go
     (doseq [component (::core/components component)]
       (<! (run-component event-channel
                          execution-state
                          component))))))

(defmethod run-component
  ::core/before-all
  ([event-channel execution-state component]
   (async/go
     ((::core/function component)))))

(defmethod run-component
  ::core/after-all
  ([event-channel execution-state component]
   (async/go
     ((::core/function component)))))

(defmethod run-component
  ::core/context
  ([event-channel execution-state component]
   (async/go
     (let [execution-state (-> execution-state
                               (with-pending-information component)
                               (with-asynchronous-information component)
                               (with-composed-around-each-functions component)
                               (with-composed-environment component))]
       (<! (run-component event-channel
                          execution-state
                          (::core/before-all component)))
       (>! event-channel
           #::{:type ::context-started
               :subject (::core/subject component)
               :current-time-millis (platform/current-time-millis)})
       (<! (run-component event-channel
                          execution-state
                          (::core/sub-components component)))
       (>! event-channel
           #::{:type ::context-finished
               :subject (::core/subject component)
               :current-time-millis (platform/current-time-millis)})
       (<! (run-component event-channel
                          execution-state
                          (::core/after-all component)))))))

(defmethod run-component
  ::core/test
  ([event-channel original-execution-state component]
   (let [execution-state (-> original-execution-state
                             (with-pending-information component)
                             (with-asynchronous-information component))
         pending? (::pending? execution-state)
         asynchronous? (::asynchronous? execution-state)
         synchronous? (not asynchronous?)]
     (async/go
       (cond pending?
             (>! event-channel
                 #::{:type ::test-marked-pending
                     :definition-line-number (::core/definition-line-number component)
                     :description (::core/description component)
                     :pending-reason (::pending-reason execution-state)
                     :current-time-millis (platform/current-time-millis)})

             synchronous?
             (let [around-each (::around-each execution-state)
                   test-function (::core/function component)
                   assertion-event-atom (atom [])
                   assertion-function (create-assertion-function assertion-event-atom)]
               (>! event-channel
                   #::{:type ::test-started
                       :namespace-name (::core/namespace-name component)
                       :definition-line-number (::core/definition-line-number component)
                       :description (::core/description component)
                       :current-time-millis (platform/current-time-millis)})
               (try (around-each (fn execute
                                   ([]
                                    (execute (fn [])))
                                   ([after-function]
                                    (try (test-function (merge ((::environment execution-state) {})
                                                               {:coconut.alpha/assert-that assertion-function
                                                                :coconut.alpha/done #(throw (platform/exception
                                                                                              (str "The done was invoked but the test is not marked asynchronous." \newline
                                                                                                   "Be sure to specify {:asynchronous {:timeout x}} in either the test" \newline
                                                                                                   "metadata or the metadata of the context which contains the test.")))}))
                                         (finally (after-function))))))
                    (<! (copy-to-channel event-channel @assertion-event-atom))
                    (>! event-channel
                        #::{:type ::test-finished
                            :description (::core/description component)
                            :current-time-millis (platform/current-time-millis)})
                    (catch #?(:clj Throwable :cljs :default) e
                      (<! (copy-to-channel event-channel @assertion-event-atom))
                      (>! event-channel
                          #::{:type ::test-threw-exception
                              :description (::core/description component)
                              :exception e
                              :current-time-millis (platform/current-time-millis)}))))

             asynchronous?
             (let [around-each (::around-each execution-state)
                   execution-finished-channel (async/chan)
                   expected-ret (platform/generate-uuid)
                   f (fn execute
                       ([]
                        (execute (fn [])))
                       ([after-function]
                        (let [assertion-event-atom (atom [])
                              done-channel (async/chan)
                              timeout-channel (async/timeout (::timeout-in-milliseconds execution-state))
                              test-function (::core/function component)
                              assertion-function (create-assertion-function assertion-event-atom)
                              done-function (create-done-function done-channel)]
                          (async/go
                            (>! event-channel
                                #::{:type ::test-started
                                    :namespace-name (::core/namespace-name component)
                                    :definition-line-number (::core/definition-line-number component)
                                    :description (::core/description component)
                                    :current-time-millis (platform/current-time-millis)})
                            (try (test-function (merge ((::environment execution-state) {})
                                                       {:coconut.alpha/assert-that assertion-function
                                                        :coconut.alpha/done done-function}))
                                 (catch #?(:clj Throwable :cljs :default) e
                                   (done-function e)))
                            (let [result (async/alts! [done-channel timeout-channel])
                                  completion-value (first result)
                                  completion-channel (last result)
                                  timed-out? (= timeout-channel completion-channel)
                                  exception-was-thrown? (platform/exception? completion-value)
                                  completed? (not (or timed-out?
                                                      exception-was-thrown?))]
                              (<! (copy-to-channel event-channel @assertion-event-atom))
                              (cond completed?
                                    (>! event-channel
                                        #::{:type ::test-finished
                                            :description (::core/description component)
                                            :current-time-millis (platform/current-time-millis)})

                                    timed-out?
                                    (>! event-channel
                                        #::{:type ::test-timed-out
                                            :description (::core/description component)
                                            :current-time-millis (platform/current-time-millis)})

                                    exception-was-thrown?
                                    (>! event-channel
                                        #::{:type ::test-threw-exception
                                            :description (::core/description component)
                                            :exception completion-value
                                            :current-time-millis (platform/current-time-millis)}))
                              (after-function)
                              (async/close! execution-finished-channel)))
                          expected-ret)))
                   actual-ret (around-each f)]
               (<! execution-finished-channel)
               (when-not (= expected-ret actual-ret)
                 (>! event-channel
                     #::{:type ::warning
                         :message "A context is marked asynchronous but there is an around-each component which assumes it will run synchronously."
                         :namespace-name (::core/namespace-name component)
                         :test-definition-line-number (::core/definition-line-number component)
                         :around-each-definition-line-number 2220
                         :current-time-millis (platform/current-time-millis)}))))))))

(defn run
  ([query-result]
   (let [event-channel (async/chan)
         execution-state #::{:asynchronous? false
                             :timeout-in-milliseconds nil
                             :pending? false
                             :pending-reason nil
                             :around-each #(%)
                             :environment identity}]
     (async/go
       (>! event-channel
           #::{:type ::suite-started
               :current-time-millis (platform/current-time-millis)
               :total-number-of-tests (::query/total-number-of-tests query-result)})
       (<! (run-component event-channel
                          execution-state
                          query-result))
       (>! event-channel
           #::{:type ::suite-finished
               :current-time-millis (platform/current-time-millis)})
       (async/close! event-channel))
     event-channel)))
