(ns coconut.alpha.core
  (:require
    [coconut.alpha.platform :as cap]
    ))

(defn asynchronous?
  ([options]
   (when (contains? options :asynchronous)
     (boolean (:asynchronous options)))))

(defn timeout-in-milliseconds
  ([options]
   (:timeout (:asynchronous options))))

(defn pending?
  ([options]
   (when (contains? options :pending)
     (boolean (:pending options)))))

(defn pending-reason
  ([options]
   (when (string? (:pending options))
     (:pending options))))

(defn tags
  ([options]
   (set (:tags options))))

(defn with-denormalized-options
  ([component options]
   (merge component
          options
          #::{:asynchronous? (asynchronous? options)
              :timeout-in-milliseconds (timeout-in-milliseconds options)
              :pending? (pending? options)
              :pending-reason (pending-reason options)
              :tags (tags options)})))

(defn add-sub-component
  ([context component]
   (let [component-type (::component-type component)]
     (cond (or (= ::context component-type)
               (= ::test component-type))
           (update-in context
                      [::sub-components ::components] conj component)

           (= ::collection component-type)
           (add-sub-component context
                              (::components component))

           (or (= ::before-all component-type)
               (= ::after-all component-type)
               (= ::around-each component-type)
               (= ::environment component-type))
           (update-in context
                      [component-type ::components] conj component)

           (sequential? component)
           (reduce add-sub-component
                   context
                   component)

           :else
           (throw (cap/illegal-argument-exception
                    (str "unsupported component: " (pr-str component))))))))

(defn add-sub-components
  ([context components]
   (reduce add-sub-component context components)))

(defn user-defined-component?
  ([object]
   (or (boolean (::component-type object))
       (sequential? object))))

(defn options-and-components
  ([arguments]
   (as-> arguments __
     (remove var? __)
     (remove nil? __)
     (split-with (complement user-defined-component?) __)
     (update __ 0 (partial reduce merge)))))

(defn string-label
  ([component]
   (let [value (case (::component-type component)
                 ::context (::subject component)
                 ::test (::description component))]
     (if (string? value)
       value
       (pr-str value)))))

(defn create-collection-component
  ([]
   (create-collection-component []))
  ([components]
   #::{:component-type ::collection
       :components components}))

(defn create-environment-component
  ([f]
   #::{:component-type ::environment
       :function f}))

(defn create-around-each-component
  ([definition-line-number f]
   #::{:component-type ::around-each
       :definition-line-number definition-line-number
       :function f}))

(defn create-after-all-component
  ([f]
   #::{:component-type ::after-all
       :function f}))

(defn create-after-each-component
  ([line-number f]
   (create-around-each-component line-number
                                 #(% f))))

(defn create-before-each-component
  ([line-number f]
   (create-around-each-component line-number
                                 #(do (f) (%)))))

(defn create-before-all-component
  ([f]
   #::{:component-type ::before-all
       :function f}))

(defn create-context-component
  ([file-name namespace-name definition-line-number subject & arguments]
   (let [[options components] (options-and-components arguments)]
     (-> #::{:component-type ::context
             :id (cap/generate-uuid)
             :file-name file-name
             :namespace-name namespace-name
             :definition-line-number definition-line-number
             :subject subject
             :before-all (create-collection-component)
             :after-all (create-collection-component)
             :around-each (create-collection-component)
             :environment (create-collection-component)
             :sub-components (create-collection-component)}
         (with-denormalized-options options)
         (add-sub-components components)))))

(defn create-test-component
  ([file-name namespace-name definition-line-number description options function]
   (-> #::{:component-type ::test
           :id (cap/generate-uuid)
           :file-name file-name
           :namespace-name namespace-name
           :definition-line-number definition-line-number
           :description description
           :function function}
       (with-denormalized-options options))))
