(ns coconut.alpha.matchers
  #?(:cljs (:require-macros [coconut.alpha.matchers]))
  (:require
    [clojure.pprint :as pp]
    [clojure.set :as cs]
    [clojure.string :as cstr]
    [coconut.alpha.platform :as platform]
    [coconut.alpha.rendering :as car]
    ))

(defn render
  ([form dispatch]
   (-> form
       (pp/write :dispatch dispatch
                 :pretty true)
       (with-out-str)
       (cstr/split #"\n"))))

(defn render-data
  ([form]
   (render form
           pp/simple-dispatch)))

(defn render-code
  ([form]
   (render form
           pp/code-dispatch)))

(defn multi-or-single-line-message
  ([options]
   (let [render-fn (::render-fn options render-data)
         rendered-data (render-fn (::data options))]
     (if (< 1 (count rendered-data))
       (list* (str (::message options)
                   \…)
              rendered-data)
       (list (apply str
                    (::message options)
                    \space
                    rendered-data))))))

(defprotocol IMatcher
  (evaluate-matcher [matcher actual]))

(extend-protocol IMatcher
  #?(:clj clojure.lang.IFn
     :cljs function)
  (evaluate-matcher [matcher actual]
    (matcher actual)))

(defn create-equality-matcher
  ([line-number expected]
   (fn [actual]
     (when-not (= expected actual)
       {:coconut.matchers/line-number line-number
        :expected (render-data expected)
        :actual (render-data actual)}))))

(defn create-negated-equality-matcher
  ([line-number expected]
   (fn [actual]
     (when (= expected actual)
       {:coconut.matchers/line-number line-number
        :expected (multi-or-single-line-message
                    #::{:message "something other than"
                        :data expected})}))))

(defn create-collection-membership-matcher
  ([line-number value]
   (fn [collection]
     (when-not (contains? (set collection) value)
       {:coconut.matchers/line-number line-number
        :expected (multi-or-single-line-message
                    #::{:message "a collection containing"
                        :data value})
        :actual (render-data collection)}))))

(defn create-subset-matcher
  ([line-number elements]
   (fn [coll]
     (when-not (cs/subset? (set elements) (set coll))
       {:coconut.matchers/line-number line-number
        :expected (multi-or-single-line-message
                    #::{:message "a collection containing"
                        :data elements})
        :actual (render-data coll)}))))

#?(:clj
(defmacro create-satisfies-predicate-matcher
  ([line-number predicate]
   `(fn [value#]
      (when-not (~predicate value#)
        {:coconut.matchers/line-number ~line-number
         :expected (multi-or-single-line-message
                     #::{:message "a value which satisfies"
                         :data '~predicate
                         :render-fn render-code})
         :actual (render-data value#)})))))

(defn create-throws-exception-matcher
  ([line-number]
   (create-throws-exception-matcher line-number platform/throwable-type))
  ([line-number exception-class]
   (create-throws-exception-matcher line-number exception-class nil))
  ([line-number exception-class exception-message]
   (fn [f]
     (try (f)
          {:coconut.matchers/line-number line-number
           :expected [(str "a " (platform/exception-class-name exception-class) " to be thrown")]
           :actual ["no exception"]}
          (catch #?(:clj Throwable :cljs :default) t
            (if-not (instance? exception-class t)
              {:coconut.matchers/line-number line-number
               :expected [(str "a " (platform/exception-class-name exception-class) " to be thrown")]
               :actual (render-data t)}
              (when exception-message
                (when (not= (platform/exception-message t) exception-message)
                  {:coconut.matchers/line-number line-number
                   :expected [(str "an exception message of " (pr-str exception-message))]
                   :actual (render-data t)}))))))))

(defn create-regex-matcher
  ([line-number regex]
   (fn [string]
     (when-not (and (string? string)
                    (re-matches regex string))
       {:coconut.matchers/line-number line-number
        :expected (multi-or-single-line-message
                    #::{:message "a string matching"
                        :data (render-data regex)})
        :actual (render-data string)}))))

(defn render-expected-for-and-or-matcher
  ([heading failures]
   (into (vector heading)
         (comp (map :expected)
               (mapcat (fn [lines]
                         (list* (str car/bullet " " (first lines))
                                (into (vector)
                                      (map (partial str "  "))
                                      (rest lines))))))
         failures)))

(defn create-or-matcher
  ([line-number matcher-one matcher-two & matchers]
   (fn [value]
     (let [matchers (list* matcher-one matcher-two matchers)
           failures (into (vector)
                          (comp (map #(% value))
                                (remove nil?))
                          matchers)]
       (when (= (count matchers)
                (count failures))
         {:coconut.matchers/line-number line-number
          :expected (render-expected-for-and-or-matcher "one of..." failures)
          :actual (render-data value)})))))

(defn create-and-matcher
  ([line-number matcher-one matcher-two & matchers]
   (fn [value]
     (let [matchers (list* matcher-one matcher-two matchers)
           failures (into (vector)
                          (comp (map #(% value))
                                (remove nil?))
                          matchers)]
       (when-not (empty? failures)
         {:coconut.matchers/line-number line-number
          :expected (render-expected-for-and-or-matcher "all of..." failures)
          :actual (render-data value)})))))
