(ns coconut.alpha.html
  (:require
    [clojure.core.match :as ccm]
    [clojure.string :as cs]
    [coconut.alpha.platform :as cap]
    [coconut.alpha.query :as query]
    [coconut.alpha.rendering :as rendering]
    [coconut.alpha.running :as running]
    [coconut.alpha.summarizing :as summarizing]
    ))

(defn color-string
  ([segment]
   (case (::rendering/color segment)
     ::rendering/grey "grey"
     ::rendering/red "red"
     ::rendering/green "green"
     ::rendering/yellow "yellow"
     ::rendering/teal "teal"
     nil ""
     (throw (cap/illegal-argument-exception
              (str "unsupported color: "
                   (::rendering/color segment)))))))

(defn scroll-to-bottom
  ([]
   (.scrollTo js/window
              0
              js/document.body.scrollHeight)))

(defn create-element
  ([tag]
   (js/document.createElement (name tag))))

(defn set-inner-html
  ([element html]
   (aset element "innerHTML" html)))

(defn set-text
  ([element text]
   (set-inner-html element
                   (cs/replace text #" " "&nbsp"))))

(defn set-class
  ([element class-name]
   (aset element "className" (name class-name))))

(defn add-to
  ([child parent]
   (.appendChild parent child)))

(defn output!
  ([data]
   (ccm/match [(::rendering/type data)]
     [::rendering/newline]
     (doto (create-element :br)
       (add-to js/document.body))

     [::rendering/line]
     (let [paragraph (create-element :p)]
       (doseq [segment (::rendering/segments data)]
         (doto (create-element :span)
           (set-text (::rendering/text segment))
           (set-class (color-string segment))
           (add-to paragraph)))
       (doto paragraph
         (add-to js/document.body))
       (scroll-to-bottom))

     [::rendering/segment]
     (let [paragraph (create-element :span)]
       (doseq [segment (::rendering/segments data)]
         (doto (create-element :span)
           (set-text (::rendering/text segment))
           (set-class (color-string segment))
           (add-to paragraph)))
       (doto paragraph
         (add-to js/document.body))
       (scroll-to-bottom)))))
