(ns coconut.alpha
  "The public API for the alpha version of coconut."
  (:refer-clojure :exclude [let for or and])
  #?(:cljs (:require-macros [coconut.alpha]))
  (:require
    [coconut.alpha.core :as core]
    [coconut.alpha.main :as main]
    [coconut.alpha.matchers :as matchers]
    [coconut.alpha.platform :as platform]
    [coconut.alpha.specs]
    ))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :matchers} is
  "Verifies the expected and actual values are equal.
  Compared using #'clojure.core/=.

  Example:
  (assert-that 42 (is 42))"
  ([expected]
   `(matchers/create-equality-matcher ~(:line (meta &form))
                                      ~expected))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :matchers} is-not
  "Verifies the expected and actual values are not equal.
  Compared using #'clojure.core/=.

  Example:
  (assert-that 42 (is 42))"
  ([value]
   `(matchers/create-negated-equality-matcher ~(:line (meta &form))
                                              ~value))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :matchers} contains-value
  "Verifies the collection contains the expected value. Collections
  are coerced into a set.

  Example:
  (assert-that [42] (contains-value 42))"
  ([value]
   `(matchers/create-collection-membership-matcher ~(:line (meta &form))
                                                   ~value))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :matchers} contains
  "Verifies the provided collection contains all expected values.
  The collection and the sequence of expected items are compared
  using #'clojure.set/subset?.

  Example:
  (assert-that [1 2 3] (contains [2]))"
  ([elements]
   `(matchers/create-subset-matcher ~(:line (meta &form))
                                    ~elements))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :matchers} satisfies
  "Verifies the value satisifes the given predicate.

  Example:
  (assert-that 42 (satisfies number?))"
  ([predicate]
   `(matchers/create-satisfies-predicate-matcher ~(:line (meta &form))
                                                 ~predicate))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :matchers} throws
  "When invoked with no arguments verifies the given functions throws
  any type of exception. When a particular exception class is given,
  verifies the given function throws an exception of that type or any
  subtype. Finally, when invoked with a class and a message, verifies
  an exception of the correct type with the expected message is thrown.
  Comparison of exception class is done using #'clojure.core/instance?.

  Example:
  (let [f #(throw (Exception. \"ouch\"))]
    (assert-that f (throws))
    (assert-that f (throws Exception))
    (assert-that f (throws Exception \"ouch\")))"
  ([]
   `(matchers/create-throws-exception-matcher ~(:line (meta &form))))
  ([exception-class]
   `(matchers/create-throws-exception-matcher ~(:line (meta &form))
                                              ~exception-class))
  ([exception-class exception-message]
   `(matchers/create-throws-exception-matcher ~(:line (meta &form))
                                              ~exception-class
                                              ~exception-message))))


#?(:clj
(defmacro ^{:coconut.alpha.api/category :matchers} matches
  "Verifies a string matches the given regular expression.

  Example:
  (assert-that \"hello\" (matches #\"^h.*$\"))"
  ([regex]
   `(matchers/create-regex-matcher ~(:line (meta &form))
                                   ~regex))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :matchers} or
  "Higher order matcher which must satisfy any of the
  given matchers.

  Example:
  (assert-that 0 (c/or (c/is 0)
                       (c/is 1)))"
  ([matcher-one matcher-two & matchers]
   `(matchers/create-or-matcher ~(:line (meta &form))
                                ~matcher-one
                                ~matcher-two
                                ~@matchers))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :matchers} and
  "Higher order matcher which must satisfy all of the
  given matchers.

  Example:
  (assert-that 0 (c/and (c/satisfies number?)
                        (c/is-not -1)))"
  ([matcher-one matcher-two & matchers]
   `(matchers/create-and-matcher ~(:line (meta &form))
                                 ~matcher-one
                                 ~matcher-two
                                 ~@matchers))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :api} let
  "The same as #'clojure.core/let but lets multiple tests be defined.
  Note that the bindings will be evaluated when the test is defined so to
  avoid shared state between tests it's important that you only bind
  immutable values. Also, the bindings will be setup long before any
  befores/afters/arounds. If you need to share data which depends on some
  setup you'll need to use #'coconut.alpha/environment.

  Example:
  (let [x 42]
    (it \"has access to x\"
      (fn [{:keys [:coconut.alpha/assert-that]}]
        ...))

    (it \"also has access to x\"
      (fn [{:keys [:coconut.alpha/assert-that]}]
        ...)))"
  ([bindings & components]
   `(clojure.core/let [~@bindings] [~@components]))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :api} for
  "The same as #'clojure.core/for but lets multiple tests be defined.

  Example:
  (for [x (range 100)]
    (it \"will be defined 100 times with a different binding of x\"
      (fn [{:keys [:coconut.alpha/assert-that]}]
        ...)))"
  ([bindings & components]
   `(clojure.core/for [~@bindings] [~@components]))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :api} it
  "Creates a test. Tests take a description of what is currently
  being tested, a map of options, and a test function which takes a
  single environment map. When tests are synchronous the environment
  map will contain an assertion funtion bound to :coconut.alpha/assert-that,
  and any keys which have been added via #'coconut.alpha/environment.
  When tests are asynchronous the environment map will also contain a
  done function bound to :coconut.alpha/done. This function either accepts
  zero arguments, or a single throwable.

  Example:
  (it \"is synchronous\"
    (fn [{:keys [:coconut.alpha/assert-that]}]
      (assert-that 42 (is 42))))

  (it \"is asynchronous\"
    {:asynchronous {:timeout 250}}
    (fn [{:keys [:coconut.alpha/assert-that
                 :coconut.alpha/done]}]
      (assert-that 42 (is 42))
      (done)))"
  ([description function]
   `(core/create-test-component ~*file*
                                ~(str *ns*)
                                ~(:line (meta &form))
                                ~description
                                {}
                                ~function))
  ([description options function]
   `(core/create-test-component ~*file*
                                ~(str *ns*)
                                ~(:line (meta &form))
                                ~description
                                ~options
                                ~function))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :api} context
  "Creates a test context. Contexts can be used to group tests which
  have similar setup and/or test similar aspects of an entity.

  Example:
  (context \"foo\"
    (it \"is related to foo\"
      (fn [{:keys [:coconut.alpha/assert-that]}] ...))

    (it \"is also related to foo\"
      (fn [{:keys [:coconut.alpha/assert-that]}] ...)))"
  ([& options-and-components]
   `(core/create-context-component ~*file*
                                   ~(str *ns*)
                                   ~(:line (meta &form))
                                   ~@options-and-components))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :api} before-all
  "Runs the given zero-arity function once before all tests in
  the containing context.

  Example:
  (context \"foo\"
    (before-all (fn [] ...))

    (it \"has the before all function run before it\"
      (fn [{:keys [:coconut.alpha/assert-that]}]
        ...)))"
  ([function]
   `(core/create-before-all-component ~function))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :api} before-each
  "Runs the given zero-arity function before each test in the
  containing context and any child contexts.

  Example:
  (context \"foo\"
    (before-each (fn [] ...))

    (it \"has the before each function run before it\"
      (fn [{:keys [:coconut.alpha/assert-that]}]
        ...)))"
  ([function]
   `(core/create-before-each-component ~(:line (meta &form))
                                       ~function))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :api} after-each
  "Runs the given zero-arity function after each test in the
  containing context and any child contexts.

  Example:
  (context \"foo\"
    (after-each (fn [] ...))

    (it \"has the after each function run after it\"
      (fn [{:keys [:coconut.alpha/assert-that]}]
        ...)))"
  ([function]
   `(core/create-after-each-component ~(:line (meta &form))
                                      ~function))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :api} after-all
  "Runs the given zero-arity function once after all tests in
  the containing context.

  Example:
  (context \"foo\"
    (after-all (fn [] ...))

    (it \"has the after all function run after it\"
      (fn [{:keys [:coconut.alpha/assert-that]}]
        ...)))"
  ([function]
   `(core/create-after-all-component ~function))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :api} around-each
  "Yields execution of the test to the given function. The function can perform
  setup, invoke the test, and then optionally provide a callback to run after
  the test has completed.

  Example:
  (context \"foo\"
    (around-each (fn [it]
                   (perform-setup)
                   (it (fn []
                         (perform-teardown)))))

    (it \"has the setup performed and will perform the teardown upon completion\"
      (fn [{:keys [:coconut.alpha/assert-that]}]
        ...)))"
  ([function]
   `(core/create-around-each-component ~(:line (meta &form))
                                       ~function))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :api} environment
  "Defines an environment component. This component takes a function
  which takes a map of data and returns an upated map. The map will be
  created immediately before executing the test function after all
  befores have been run.

  Example:
  (context \"foo\"
    (environment (fn [e]
                   (assoc e :x 42)))

    (it \"can access the additional environment data\"
      (fn [{:keys [:coconut.alpha/assert-that
                   :x]}]
        ...)))"
  ([function]
   `(core/create-environment-component ~function))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :api} describe
  "Defines a top-level test context. Describe components should
  not be nested. Use #'coconut.alpha/context for nested contexts.

  Example:
  (describe #'core/foo
    (it \"does something\"
      (fn [{:keys [:coconut.alpha/assert-that]}] ...)))"
  ([& options-and-components]
   `(-> (core/create-context-component ~*file*
                                       ~(str *ns*)
                                       ~(:line (meta &form))
                                       ~@options-and-components)
        (platform/register-component ~(str *ns*)
                                     {::platform/component-version 1})))))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :api} deftest
  "Defines a single top-level test.

  Example:
  (deftest \"+ returns the sum of two numbers\"
    (fn [{:keys [:coconut.alpha/assert-that]}]
      (assert-that (+ 1 2) (c/is 3)))))"
  ([description function]
   `(-> (core/create-test-component ~*file*
                                    ~(str *ns*)
                                    ~(:line (meta &form))
                                    ~description
                                    {}
                                    ~function)
        (platform/register-component ~(str *ns*)
                                     {::platform/component-version 1})))
  ([description options function]
   `(-> (core/create-test-component ~*file*
                                    ~(str *ns*)
                                    ~(:line (meta &form))
                                    ~description
                                    ~options
                                    ~function)
        (platform/register-component ~(str *ns*)
                                     {::platform/component-version 1})))))

(defn ^{:export true
        :coconut.alpha.api/category :main} run
  "Runs the test suite according to the given options. Options should
  be a map containing the following keys:

  :reporters -- A vector of reporters to use. Possible options are...
  ------------------------------------------------------------------------------------------
               :progress | Renders single-character results for a test similar
                         | to RSpec.
                         |
          :documentation | Renders the test description for each test as it
                         | finishes similar to RSpec.
                         |
                :results | Renders the final report of a test run as well as
                         | infomration about failing tests, pending tests, tests
                         | which threw an exception, and asynchronous tests which
                         | timed out.

  :output -- Where to output the rendered test reports. Possible options are...
  ------------------------------------------------------------------------------------------
                    :cli | Prints the rendered test report data to the terminal.
                         |
                   :html | Appends styled HTML elements to the DOM. (ClojureScript only).
                         |
        :browser-console | Outputs colored browser console messages. (ClojureScript only).

  :criteria -- The query used to determine which tests to run. Possible options are...
  ------------------------------------------------------------------------------------------
                    :all | Matches any defined test
                         |
       :within-namespace | Matches any test defined within the given namespace.
                         | Namespace here refers to one of three possible things...
                         |   1.) The string name of the namespace.
                         |   2.) The symbol name of the namespace.
                         |   3.) The actual namespace object (Clojure only).
                         |
      :within-namespaces | Matches any test defined in any of the given namespaces.
                         | See above for the possible types of the namespace arguments.
                         |
                    :not | Negates the given criteria.
                         |
                     :or | Combines two criteria using logical OR.
                         |
                    :and | Combines two criteria using logical AND.
                         |
        :defined-on-line | Returns any test which is defined on a specific line.
                         | Typically this will be used in conjunction with something like
                         | :within-namespace to run a certain test.
                         |
    :description-matches | Matches any test whose full description matches the given
                         | regex. The description will be the full test description
                         | which is the text defined on the test itself prepended with
                         | the text of each context within which it is defined.
                         |
      :namespace-matches | Matches any test whose namespaces matches the given regex.
                         |
                :has-tag | Matches any test which contains the given tag. Tags are matched
                         | using the the standard #'clojure.core/=.

  Example:
  (run {:reporters [:progress :results]
        :output :cli
        :criteria [:and
                   [:namespace-matches #\"foo.*-test\"]
                   [:not [:or [:has-tag :integration]
                              [:has-tag :slow]]]]})

  (run {:reporters [:documentation :results]
        :output :html
        :criteria [:all]})"
  ([options]
   (main/run options)))

#?(:clj
(defmacro ^{:coconut.alpha.api/category :main} explore
  "Describes the public API for the 'coconut.alpha namespace."
  ([]
   `(main/explore))))
