(ns coconut.alpha.reporting
  (:require
    [clojure.core.match :as ccm]
    [clojure.string :as cs]
    [coconut.alpha.aggregation :as caa]
    [coconut.alpha.platform :as cap]
    [coconut.alpha.rendering :as car]
    [coconut.alpha.summarizing :as cas]
    ))

(defn with-indicies
  ([elements]
   (into (vector)
         (comp (map-indexed vector)
               (map #(update % 0 inc)))
         elements)))

(defn progress
  ([state event]
   (ccm/match [(::cas/type event)]
     [::cas/suite-started]
     #::caa{:events []
            :state #::{:total-number-of-tests (::cas/total-number-of-tests event)
                       :pending-tests []
                       :failing-tests []
                       :level 0}}

     [::cas/context-started]
     #::caa{:events []
            :state (update state
                           ::level inc)}

     [::cas/context-finished]
     #::caa{:events []
            :state (update state
                           ::level dec)}

     [::cas/test-pending]
     #::caa{:events [#::car{:type ::car/segment
                            :segments [#::car{:text car/asterisk
                                              :color ::car/yellow}]}]
            :state (update state
                           ::pending-tests conj event)}

     [::cas/test-passed]
     #::caa{:events [#::car{:type ::car/segment
                            :segments [#::car{:text car/dot
                                              :color ::car/green}]}]
            :state state}

     [(:or ::cas/test-failed
           ::cas/test-threw-exception
           ::cas/test-timed-out)]
     #::caa{:events [#::car{:type ::car/segment
                            :segments [#::car{:text car/F
                                              :color ::car/red}]}]
            :state (update state
                           ::failing-tests conj event)}

     [::cas/suite-finished]
     #::caa{:events [#::car{:type ::car/newline}]
            :state nil})))

(defn documentation
  ([state event]
   (ccm/match [(::cas/type event)]
     [::cas/suite-started]
     #::caa{:events []
            :state #::{:total-number-of-tests (::cas/total-number-of-tests event)
                       :pending-tests []
                       :failing-tests []
                       :level 0}}

     [::cas/context-started]
     #::caa{:events [#::car{:type ::car/line
                            :segments [#::car{:text (car/indent (::level state))}
                                       #::car{:text (car/render-subject (::cas/subject event))}]}]
            :state (update state
                           ::level inc)}

     [::cas/context-finished]
     #::caa{:events []
            :state (update state
                           ::level dec)}

     [::cas/test-pending]
     #::caa{:events [#::car{:type ::car/line
                            :segments [#::car{:text (car/indent (::level state))}
                                       #::car{:text car/asterisk
                                              :color ::car/yellow}
                                       #::car{:text (car/spacing 1)}
                                       #::car{:text (::cas/description event)
                                              :color ::car/yellow}]}]
            :state (update state
                           ::pending-tests conj event)}

     [::cas/test-passed]
     #::caa{:events [#::car{:type ::car/line
                            :segments [#::car{:text (car/indent (::level state))}
                                       #::car{:text car/checkmark
                                              :color ::car/green}
                                       #::car{:text (car/spacing 1)}
                                       #::car{:text (::cas/description event)
                                              :color ::car/green}]}]
            :state state}

     [(:or ::cas/test-failed
           ::cas/test-threw-exception
           ::cas/test-timed-out)]
     #::caa{:events [#::car{:type ::car/line
                            :segments [#::car{:text (car/indent (::level state))}
                                       #::car{:text car/x
                                              :color ::car/red}
                                       #::car{:text (car/spacing 1)}
                                       #::car{:text (::cas/description event)
                                              :color ::car/red}]}]
            :state (update state
                           ::failing-tests conj event)}

     [::cas/suite-finished]
     #::caa{:events []
            :state state})))

(defn results
  ([state event]
   (ccm/match [(::cas/type event)]
     [::cas/suite-started]
     #::caa{:events []
            :state #::{:total-number-of-tests (::cas/total-number-of-tests event)
                       :pending-tests []
                       :failing-tests []
                       :level 0}}

     [::cas/context-started]
     #::caa{:events []
            :state (update state
                           ::level inc)}

     [::cas/context-finished]
     #::caa{:events []
            :state (update state
                           ::level dec)}

     [::cas/test-pending]
     #::caa{:events []
            :state (update state
                           ::pending-tests conj event)}

     [::cas/test-passed]
     #::caa{:events []
            :state state}

     [(:or ::cas/test-failed
           ::cas/test-threw-exception
           ::cas/test-timed-out)]
     #::caa{:events []
            :state (update state
                           ::failing-tests conj event)}

     [::cas/suite-finished]
     (let [put! (fn [a e]
                  (swap! a conj e))
           events (atom [])]
       (when (seq (::pending-tests state))
         (put! events
               #::car{:type ::car/newline})
         (put! events
               #::car{:type ::car/line
                      :segments [#::car{:text "Pending Tests..."}]})
         (doseq [[n t] (with-indicies (::pending-tests state))]
           (let [list-index (car/render-list-index n)]
             (put! events
                   #::car{:type ::car/newline})
             (let [description-lines (car/render-full-description (::cas/context t)
                                                                  (::cas/description t))]
               (put! events
                     #::car{:type ::car/line
                            :segments [#::car{:text (car/indent 1)}
                                       #::car{:text list-index}
                                       #::car{:text (car/spacing 1)}
                                       #::car{:text (first description-lines)}]})
               (doseq [line (rest description-lines)]
                 (put! events
                       #::car{:type ::car/line
                              :segments [#::car{:text (car/indent 1)}
                                         #::car{:text (car/spacing (count list-index))}
                                         #::car{:text (car/spacing 1)}
                                         #::car{:text line}]})))
             (put! events
                   #::car{:type ::car/line
                          :segments [#::car{:text (car/indent 1)}
                                     #::car{:text (car/spacing (count list-index))}
                                     #::car{:text (car/spacing 1)}
                                     #::car{:text (car/pending-reason (::cas/pending-reason t))
                                            :color ::car/yellow}]}))))

       (when (seq (::failing-tests state))
         (put! events
               #::car{:type ::car/newline})
         (put! events
               #::car{:type ::car/line
                      :segments [#::car{:text "Failures..."}]})
         (doseq [[n t] (with-indicies (::failing-tests state))]
           (let [list-index (car/render-list-index n)]
             (put! events
                   #::car{:type ::car/newline})
             (let [description-lines (car/render-full-description (::cas/context t)
                                                                  (::cas/description t))]
               (put! events
                     #::car{:type ::car/line
                            :segments [#::car{:text (car/indent 1)}
                                       #::car{:text list-index}
                                       #::car{:text (car/spacing 1)}
                                       #::car{:text (first description-lines)}]})
               (doseq [line (rest description-lines)]
                 (put! events
                       #::car{:type ::car/line
                              :segments [#::car{:text (car/indent 1)}
                                         #::car{:text (car/spacing (count list-index))}
                                         #::car{:text (car/spacing 1)}
                                         #::car{:text line}]})))
             (put! events
                   #::car{:type ::car/line
                          :segments [#::car{:text (car/indent 1)}
                                     #::car{:text (car/spacing (count list-index))}
                                     #::car{:text (car/spacing 2)}
                                     #::car{:text "namespace:"
                                            :color ::car/teal}
                                     #::car{:text (car/spacing 1)}
                                     #::car{:text (::cas/namespace-name t)
                                            :color ::car/red}]})
             (put! events
                   #::car{:type ::car/line
                          :segments [
                                     #::car{:text (car/indent 1)}
                                     #::car{:text (car/spacing (count list-index))}
                                     #::car{:text (car/spacing 7)}
                                     #::car{:text "line:"
                                            :color ::car/teal}
                                     #::car{:text (car/spacing 1)}
                                     #::car{:text (str (::cas/line-number t))
                                            :color ::car/red}]})
             (case (::cas/type t)
               ::cas/test-failed
               (do (put! events
                         #::car{:type ::car/line
                                :segments [#::car{:text (car/indent 1)}
                                           #::car{:text (car/spacing (count list-index))}
                                           #::car{:text (car/spacing 3)}
                                           #::car{:text "expected:"
                                                  :color ::car/teal}
                                           #::car{:text (car/spacing 1)}
                                           #::car{:text (first (::cas/expected t))
                                                  :color ::car/red}]})
                   (doseq [line (rest (::cas/expected t))]
                     (put! events
                           #::car{:type ::car/line
                                  :segments [#::car{:text (car/indent 1)}
                                             #::car{:text (car/spacing (count list-index))}
                                             #::car{:text (car/spacing 13)}
                                             #::car{:text line
                                                    :color ::car/red}]}))
                   (when-let [actual (::cas/actual t)]
                     (put! events
                           #::car{:type ::car/line
                                  :segments [#::car{:text (car/indent 1)}
                                             #::car{:text (car/spacing (count list-index))}
                                             #::car{:text (car/spacing 5)}
                                             #::car{:text "actual:"
                                                    :color ::car/teal}
                                             #::car{:text (car/spacing 1)}
                                             #::car{:text (first (::cas/actual t))
                                                    :color ::car/red}]})
                     (doseq [line (rest (::cas/actual t))]
                       (put! events
                             #::car{:type ::car/line
                                    :segments [#::car{:text (car/indent 1)}
                                               #::car{:text (car/spacing (count list-index))}
                                               #::car{:text (car/spacing 13)}
                                               #::car{:text line
                                                      :color ::car/red}]}))))

               ::cas/test-threw-exception
               (let [e (::cas/exception t)
                     stack-trace (cap/exception-stack-trace e)]
                 (put! events
                       #::car{:type ::car/line
                              :segments [#::car{:text (car/indent 1)}
                                         #::car{:text (car/spacing (count list-index))}
                                         #::car{:text (car/spacing 2)}
                                         #::car{:text "exception:"
                                                :color ::car/teal}
                                         #::car{:text (car/spacing 1)}
                                         #::car{:text (cap/exception-class-name e)
                                                :color ::car/red}]})
                 (let [label "message:"
                       lines (cs/split (or (cap/exception-message e)
                                           "-")
                                       #"\n")]
                   (put! events
                         #::car{:type ::car/line
                                :segments [#::car{:text (car/indent 1)}
                                           #::car{:text (car/spacing (count list-index))}
                                           #::car{:text (car/spacing 4)}
                                           #::car{:text label
                                                  :color ::car/teal}
                                           #::car{:text (car/spacing 1)}
                                           #::car{:text (first lines)
                                                  :color ::car/red}]})
                   (doseq [line (rest lines)]
                     (put! events
                           #::car{:type ::car/line
                                  :segments [#::car{:text (car/indent 1)}
                                             #::car{:text (car/spacing (count list-index))}
                                             #::car{:text (car/spacing 4)}
                                             #::car{:text (car/spacing (count label))}
                                             #::car{:text (car/spacing 1)}
                                             #::car{:text line
                                                    :color ::car/red}]})))
                 (put! events
                       #::car{:type ::car/line
                              :segments [#::car{:text (car/indent 1)}
                                         #::car{:text (car/spacing (count list-index))}
                                         #::car{:text "stack trace:"
                                                :color ::car/teal}
                                         #::car{:text (car/spacing 1)}
                                         #::car{:text (first stack-trace)
                                                :color (car/stack-trace-element-color (first stack-trace))}]})
                 (doseq [stack-trace-element (rest stack-trace)]
                   (put! events
                         #::car{:type ::car/line
                                :segments [#::car{:text (car/indent 1)}
                                           #::car{:text (car/spacing (count list-index))}
                                           #::car{:text (car/spacing 13)}
                                           #::car{:text stack-trace-element
                                                  :color (car/stack-trace-element-color stack-trace-element)}]})))

               ::cas/test-timed-out
               (do (put! events
                         #::car{:type ::car/line
                                :segments [#::car{:text (car/indent 1)}
                                           #::car{:text (car/spacing (count list-index))}
                                           #::car{:text (car/spacing 5)}
                                           #::car{:text "reason:"
                                                  :color ::car/teal}
                                           #::car{:text (car/spacing 1)}
                                           #::car{:text "TIMEOUT"
                                                  :color ::car/red}]}))

               (throw (cap/illegal-argument-exception
                        (str "unsupported cas event type: "
                             (::cas/type t))))))))
       (when-let [warnings (seq (::cas/warnings event))]
         (put! events
               #::car{:type ::car/newline})
         (put! events
               #::car{:type ::car/line
                      :segments [#::car{:text "Warnings..."}]})
         (doseq [[n t] (with-indicies warnings)]
           (let [list-index (car/render-list-index n)]
             (put! events
                   #::car{:type ::car/newline})
             (let [message-lines (car/render-full-description (::cas/message t))]
               (put! events
                     #::car{:type ::car/line
                            :segments [#::car{:text (car/indent 1)}
                                       #::car{:text list-index}
                                       #::car{:text (car/spacing 1)}
                                       #::car{:text (first message-lines)}]})
               (doseq [line (rest message-lines)]
                 (put! events
                       #::car{:type ::car/line
                              :segments [#::car{:text (car/indent 1)}
                                         #::car{:text (car/spacing (count list-index))}
                                         #::car{:text (car/spacing 1)}
                                         #::car{:text line}]})))
             (put! events
                   #::car{:type ::car/line
                          :segments [#::car{:text (car/indent 1)}
                                     #::car{:text (car/spacing (count list-index))}
                                     #::car{:text (car/spacing 2)}
                                     #::car{:text "namespace:"
                                            :color ::car/teal}
                                     #::car{:text (car/spacing 1)}
                                     #::car{:text (::cas/namespace-name t)
                                            :color ::car/red}]})
             (put! events
                   #::car{:type ::car/line
                          :segments [#::car{:text (car/indent 1)}
                                     #::car{:text (car/spacing (count list-index))}
                                     #::car{:text (car/spacing 7)}
                                     #::car{:text "line:"
                                            :color ::car/teal}
                                     #::car{:text (car/spacing 1)}
                                     #::car{:text (str (::cas/test-definition-line-number t))
                                            :color ::car/red}]}))))
       (put! events
             #::car{:type ::car/newline})
       (put! events
             #::car{:type ::car/line
                    :segments [#::car{:text (cap/format "Finished in %s seconds"
                                                        (car/render-suite-duration (::cas/duration-in-milliseconds event)))}]})
       (put! events
             #::car{:type ::car/line
                    :segments [#::car{:text (car/render-number-of-tests (::total-number-of-tests state)
                                                                        (count (::failing-tests state))
                                                                        (count (::pending-tests state)))
                                      :color (car/number-of-tests-color (count (::failing-tests state))
                                                                        (count (::pending-tests state)))}]})
       #::caa{:events @events
              :state state}))))
