(ns coconut.alpha.rendering
  (:require
    [clojure.string :as cs]
    [coconut.alpha.platform :as platform]
    ))

(def ^{:const true} checkmark \✔)
(def ^{:const true} x         \✘)
(def ^{:const true} asterisk  \*)
(def ^{:const true} dot       \.)
(def ^{:const true} F         \F)

(defn word-wrap
  ([line]
   (if (empty? line)
     []
     [line])))

(defn render-subject
  ([object]
   (if (string? object)
     object
     (pr-str object))))

(defn spacing
  ([n]
   (apply str
          (take n
                (repeat \space)))))

(defn indent
  ([n]
   (spacing (* 4 n))))

(defn stack-trace-element-color
  ([el]
   (if (re-matches platform/language-specific-stack-trace-element-regex
                   (str el))
     ::grey
     ::red)))

(defn render-suite-duration
  ([duration]
   (str (float (/ duration 1000)))))

(defn render-list-index
  ([n]
   (platform/format "%s)" n)))

(defn render-full-description
  ([description]
   (render-full-description [] description))
  ([context description]
   (let [words (cs/split (transduce (interpose \space)
                                    str
                                    (conj (vec context)
                                          description))
                         #" ")
         line (volatile! "")
         lines (volatile! [])]
     (doseq [word words]
       (let [updated-line (if (empty? @line)
                            word
                            (str @line \space word))]
         (if (< 80 (count updated-line))
           (do (vreset! line "")
               (vswap! lines conj updated-line))
           (vreset! line updated-line))))
     (if (empty? @line)
       @lines
       (conj @lines @line)))))

(defn pending-reason
  ([reason]
   (if (some? reason)
     (str "PENDING -- " reason)
     "PENDING")))

(defn render-number-of-tests
  ([total-count failing-count pending-count]
   (platform/format (if (not (zero? pending-count))
                      "%s examples, %s failures, %s pending"
                      "%s examples, %s failures")
                    total-count
                    failing-count
                    pending-count)))

(defn number-of-tests-color
  ([failing-count pending-count]
   (cond (not (zero? failing-count)) ::red
         (not (zero? pending-count)) ::yellow
         :else ::green)))
