(ns coconut.alpha.query
  (:refer-clojure :exclude [and or not])
  (:require
    [clojure.core.match :as ccm]
    [clojure.set :as cs]
    [coconut.alpha.core :as core]
    [coconut.alpha.platform :as platform]
    ))

(defn satisfied?
  ([criteria state]
   ((::predicate criteria) state)))

(defn number-of-tests
  ([component]
   (ccm/match [(::core/component-type component)]
     [(:or ::result
           ::core/context)]
     (number-of-tests (::core/sub-components component))

     [::core/collection]
     (transduce (map number-of-tests)
                +
                (::core/components component))

     [::core/test] 1)))

(defn with-current-component-type
  ([state component]
   (assoc state
          ::current-component-type (::core/component-type component))))

(defn with-merged-description
  ([state component]
   (assoc state
          ::description (if (empty? (::description state))
                          (core/string-label component)
                          (platform/format "%s %s"
                                           (::description state)
                                           (core/string-label component))))))

(defn with-namespace-name
  ([state component]
   (assoc state
          ::namespace-name (::core/namespace-name component))))

(defn with-definition-line-number
  ([state component]
   (assoc state
          ::definition-line-number (::core/definition-line-number component))))

(defn with-tags
  ([state component]
   (update state
           ::tags (partial cs/union
                           (set (::core/tags component))))))

(defn merge-component-information
  ([state component]
   (-> state
       (with-current-component-type component)
       (with-merged-description component)
       (with-namespace-name component)
       (with-definition-line-number component)
       (with-tags component))))

(declare filter-components)

(defmulti filter-component
  (fn [criteria state component]
    (::core/component-type component)))

(defmethod filter-component
  ::core/collection
  ([criteria state component]
   (update component
           ::core/components
           (partial into
                    (vector)
                    (filter-components criteria state)))))

(defmethod filter-component
  ::core/context
  ([criteria state component]
   (let [updated-state (merge-component-information state component)]
     (if (satisfied? criteria updated-state)
       component
       (update component
               ::core/sub-components
               (partial filter-component
                        criteria
                        updated-state))))))

(defmethod filter-component
  ::core/test
  ([criteria state component]
   (let [updated-state (merge-component-information state component)]
     (when (satisfied? criteria updated-state)
       component))))

(defn filter-components
  ([criteria state]
   (comp (map (partial filter-component criteria state))
         (remove nil?)
         (remove (comp zero? number-of-tests)))))

(defn build-query-result
  ([]
   #::core{:component-type ::result
           ::total-number-of-tests 0
           :sub-components (core/create-collection-component)})
  ([result component]
   (-> result
       (update ::total-number-of-tests + (number-of-tests component))
       (update-in [::core/sub-components ::core/components] conj component))))

(defn or*
  ([criteria]
   (fn [state]
     (loop [criteria criteria]
       (if criteria
         (let [criterion (first criteria)]
           (if (satisfied? criterion state)
             true
             (recur (next criteria))))
         false)))))

(defn and*
  ([criteria]
   (fn [state]
     (loop [criteria criteria]
       (if criteria
         (let [criterion (first criteria)]
           (if (satisfied? criterion state)
             (recur (next criteria))
             false))
         true)))))

(defn not
  ([criteria]
   {::predicate         (::negated-predicate criteria (complement (::predicate criteria)))
    ::negated-predicate (::predicate criteria)}))

(defn or
  ([criterion & criteria]
   (let [criteria (list* criterion criteria)]
     {::predicate         (or* criteria)
      ::negated-predicate (and* (sequence (map not) criteria))})))

(defn and
  ([criterion & criteria]
   (let [criteria (list* criterion criteria)]
     {::predicate         (and* criteria)
      ::negated-predicate (or* (sequence (map not) criteria))})))

(def all
  {::predicate (constantly true)})

(defn within-namespace
  ([namespace-name]
   {::predicate (fn [state]
                  (= (str namespace-name)
                     (::namespace-name state)))}))

(defn within-namespaces
  ([namespace-name & namespace-names]
   (transduce (map within-namespace)
              (completing or)
              (not all)
              (list* namespace-name namespace-names))))

(defn namespace-matches
  ([regex]
   {::predicate (fn [state]
                  (boolean (re-matches regex
                                       (::namespace-name state))))}))

(defn defined-on-line
  ([line-number]
   {::predicate         (fn [state]
                          (= line-number
                             (::definition-line-number state)))
    ::negated-predicate (fn [state]
                          (clojure.core/and (= ::core/test
                                               (::current-component-type state))
                                            (not= line-number
                                                  (::definition-line-number state))))}))

(defn description-matches
  ([regex]
   {::predicate         (fn [state]
                          (clojure.core/and (= ::core/test
                                               (::current-component-type state))
                                            (boolean (re-matches regex
                                                                 (::description state)))))
    ::negated-predicate (fn [state]
                          (clojure.core/and (= ::core/test
                                               (::current-component-type state))
                                            (clojure.core/not (boolean (re-matches regex
                                                                                   (::description state))))))}))

(defn has-tag
  ([tag]
   {::predicate (fn [state]
                  (contains? (::tags state) tag))}))

(defn parse
  ([query]
   (ccm/match query
     [:all] all
     [:within-namespace value] (within-namespace value)
     [:within-namespaces & values] (apply within-namespaces values)
     [:not criterion] (not (parse criterion))
     [:or & criteria] (apply or (into (vector) (map parse) criteria))
     [:and & criteria] (apply and (into (vector) (map parse) criteria))
     [:defined-on-line line-number] (defined-on-line line-number)
     [:description-matches re] (description-matches re)
     [:namespace-matches re] (namespace-matches re)
     [:has-tag tag] (has-tag tag))))

(defn query
  ([criteria]
   (query criteria
          (platform/registered-components #::platform{:component-version 1})))
  ([criteria defined-components]
   (let [state #::{:description ""
                   :definition-line-number nil
                   :namespace-name nil
                   :tags #{}}]
     (transduce (filter-components (if (map? criteria)
                                     criteria
                                     (parse criteria))
                                   state)
                (completing build-query-result)
                defined-components))))

(defn return-all
  ([collection]
   (query all collection)))

(defn return
  ([component]
   (return-all [component])))
