(ns ham-fisted.protocols
  (:require [ham-fisted.defprotocol :refer [defprotocol extend-protocol extend-type extend]:as hamf-defproto])
  (:import [clojure.lang IFn IReduceInit IDeref]
           [java.util.function DoubleConsumer]
           [java.util Map]
           [ham_fisted Sum Sum$SimpleSum Reducible IFnDef$ODO ParallelOptions
            Reductions IMutList])
  (:refer-clojure :exclude [reduce set? count defprotocol extend-protocol extend-type extend]))

(defprotocol Counted
  (^long count [m]))

(defprotocol ToIterable
  (convertible-to-iterable? [item])
  (->iterable [item]))


(defprotocol ToCollection
  (convertible-to-collection? [item])
  (->collection [item]))


(defprotocol Reduction
  "Faster check than satisfies? to see if something is reducible"
  (reducible? [coll]))


(extend-protocol Reduction
  nil
  (reducible? [this] true)
  Object
  (reducible? [this]
    (or (instance? IReduceInit this)
        (instance? Iterable this)
        (instance? Map this)
        ;;This check is dog slow
        (clojure.core/satisfies? clojure.core.protocols/CollReduce this))))


(defprotocol ParallelReduction
  "Protocol to define a parallel reduction in a collection-specific pathway.  Specializations
  are in impl as that is where the parallelization routines are found."
  (preduce [coll init-val-fn rfn merge-fn ^ParallelOptions options]
    "Container-specific parallelized reduction.  Reductions must respect the pool passed in via
the options."))


(defprotocol Finalize
  "Generic protocol for things that finalize results of reductions.  Defaults to deref of
  instance of IDeref or identity."
  (finalize [this val]))


(extend-protocol Finalize
  Object
  (finalize [this val]
    (if (instance? IDeref val)
      (.deref ^IDeref val)
      val))
  ;;clojure rfn equivalence
  IFn
  (finalize [this val]
    (this val)))


(defprotocol Reducer
  "Reducer is the basic reduction abstraction as a single object."
  (->init-val-fn [item]
    "Returns the initial values for a parallel reduction.  This function
takes no arguments and returns the initial accumulator.")
  (->rfn [item]
    "Returns the reduction function for a parallel reduction. This function takes
two arguments, the accumulator and a value from the collection and returns a new
or modified accumulator."))


(extend-protocol Reducer
  IFn
  (->init-val-fn [this] this)
  (->rfn [this] this))


(defprotocol ParallelReducer
  "Parallel reducers are simple a single object that you can pass into preduce as
  opposed to 3 separate functions."
  (->merge-fn [item]
    "Returns the merge function for a parallel reduction.  This function takes
two accumulators  and returns a or modified accumulator."))


(extend-protocol ParallelReducer
  IFn
  (->merge-fn [this] this))


(extend-protocol ParallelReducer
  Object
  (->merge-fn [this]
    (fn [_l _r] (throw (RuntimeException. (str "Object does not implement merge: "
                                               (type this)))))))


(def ^:no-doc double-consumer-accumulator
  (reify IFnDef$ODO
    (invokePrim [f acc v]
      (.accept ^DoubleConsumer acc v)
      acc)))

(defn- reducible-merge
  [^Reducible lhs rhs]
  (.reduce lhs rhs))

(defprotocol PAdd
  "Define a function to mutably add items to a collection.  This function must return
  the collection -- it must be useable in a reduce as the rf."
  (add-fn [l]))

(defprotocol SetOps
  "Simple protocol for set operations to make them uniformly extensible to new objects."
  (set? [l])
  (union [l r])
  (difference [l r])
  (intersection [l r])
  (xor [l r])
  (contains-fn [item]
    "Return an efficient function for deciding if this set contains a single item.")
  (^long cardinality [item]
   "Some sets don't work with clojure's count function."))


(defprotocol BulkSetOps
  (reduce-union [l data])
  (reduce-intersection [l data]))


(defprotocol BitSet
  "Protocol for efficiently dealing with bitsets"
  (bitset? [item])
  (contains-range? [item sidx eidx])
  (intersects-range? [item sidx eidx])
  (min-set-value [item])
  (max-set-value [item]))


(defprotocol WrapArray
  (^IMutList wrap-array [ary])
  (^IMutList wrap-array-growable [ary ptr]))


(defprotocol SerializeObjBytes
  (serialize->bytes [o]))


(defprotocol Datatype
  (datatype [o]
    "Returns the datatype [:int8, :int16, etc] -- if known --
else the type can be assumed to be an object type.  The return value may not be a keyword
but it must be comparable with identical?")
  (simplified-datatype [o]
    "Returns exactly :int64, :float64, or :object"))

(hamf-defproto/extend nil Datatype {:datatype :object
                                    :simplified-datatype :object})

(hamf-defproto/extend Object Datatype {:datatype :object
                                       :simplified-datatype :object})

(defprotocol ContainedDatatype
  (contained-datatype [o]
    "Datatype of contained datatype - may be nil if not a container")
  (simplified-contained-datatype [o]
    "Exactly :int64 :float64 :object or nil"))

(extend nil ContainedDatatype
        {:contained-datatype nil
         :simplified-contained-datatype nil})

(extend Object ContainedDatatype
        {:contained-datatype nil
         :simplified-contained-datatype nil})


(defprotocol ReturnedDatatype
  (returned-datatype [o])
  (simplified-returned-datatype [o]))

(extend nil ReturnedDatatype
        {:returned-datatype nil
         :simplified-returned-datatype nil})

(extend Object ReturnedDatatype
        {:returned-datatype nil
         :simplified-returned-datatype nil})
