(ns avclj.avcodec
  ;;Getting the context layout correct requires it's own file!!
  (:require [avclj.av-context :as av-context]
            [avclj.av-error :as av-error]
            [avclj.av-pixfmt :as av-pixfmt]
            [tech.v3.datatype.ffi :as dt-ffi]
            [tech.v3.datatype.native-buffer :as native-buffer]
            [tech.v3.datatype.errors :as errors]
            [tech.v3.resource :as resource])
  (:import [tech.v3.datatype.ffi Pointer]
           [java.util Map]))

(declare str-error)

(defmacro check-error
  [_fn-def & body]
  `(let [error-val# (long (do ~@body))]
     (errors/when-not-errorf
      (>= error-val# 0)
      "Exception calling avcodec: (%d) - \"%s\""
      error-val# (if-let [err-name#  (get av-error/value->error-map error-val#)]
                   err-name#
                   (str-error error-val#)))
     error-val#))


(dt-ffi/define-library!
  lib
  '{:avcodec_version {:rettype :int32
                      :doc "Return the version of the avcodec library"}
    :avcodec_configuration {:rettype :string
                            :doc "Return the build configuration of the avcodec lib"}
    :avcodec_license {:rettype :string
                      :doc "Return the license of the avcodec lib"}
    :av_codec_iterate {:rettype :pointer
                       :argtypes [[opaque :pointer]]
                       :doc "Iterate through av code objects.  Returns an AVCodec*"}
    :av_strerror {:rettype :int32
                  :argtypes [[errnum :int32]
                             [errbuf :pointer]
                             [errbuf-size :size-t]]
                  :doc "Get a string description for an error."}
    :av_codec_is_encoder {:rettype :int32
                          :argtypes [[codec :pointer]]
                          :doc "Return nonzero if encoder"}
    :av_codec_is_decoder {:rettype :int32
                          :argtypes [[codec :pointer]]
                          :doc "Return nonzero if encoder"}
    :avcodec_find_encoder_by_name {:rettype :pointer
                                   :argtypes [[name :string]]
                                   :doc "Find an encoder by name"}
    :avcodec_find_encoder {:rettype :pointer
                           :argtypes [[id :int32]]
                           :doc "Find an encoder codec id"}
    :avcodec_find_decoder_by_name {:rettype :pointer
                                   :argtypes [[name :string]]
                                   :doc "Find a decoder by name"}
    :avcodec_find_decoder {:rettype :pointer
                           :argtypes [[id :int32]]
                           :doc "Find a decoder codec id"}
    :avcodec_alloc_context3 {:rettype :pointer
                             :argtypes [[codec :pointer?]]
                             :doc "Allocate an avcodec context"}
    :avcodec_free_context {:rettype :void
                           :argtypes [[codec-ptr-ptr :pointer]]
                           :doc "Free the context.  Expects a ptr-ptr to be passed in"}
    :avcodec_parameters_from_context {:rettype :int32
                                      :argtypes [[par :pointer]
                                                 [codec :pointer]]
                                      :doc "Initialize parameter struct from the codec"}
    :avcodec_parameters_to_context {:rettype :int32
                                    :argtypes [[codec :pointer]
                                               [par :pointer]]
                                    :doc "Initialize codec params from param struct"}
    :avcodec_open2 {:rettype :int32
                    :argtypes [[c :pointer]
                               [codec :pointer]
                               [opts :pointer?]]
                    :check-error? true
                    :doc "Open the codec with the context"}
    :av_opt_set {:rettype :int32
                 :argtypes [[priv-data :pointer]
                            [key :string]
                            [value :string]
                            [flags :int32]]
                 :doc "Set an option on an object such as the codec context"
                 :check-error? true}
    :av_packet_alloc {:rettype :pointer
                      :doc "allocate an av packet"}
    :av_packet_free {:rettype :void
                     :argtypes [[packet :pointer]] ;;AVPacket**
                     :doc "free an av packet"}
    :av_packet_rescale_ts {:rettype :void
                           :argtypes [[pkt :pointer]
                                      [src_num :int32]
                                      [src_den :int32]
                                      [dst_num :int32]
                                      [dst_den :int32]]
                           :doc "rescale a packet's time fields"}
    :av_frame_alloc {:rettype :pointer
                     :doc "allocate an av frame"}
    :av_frame_free {:rettype :void
                    :argtypes [[frame :pointer]]
                    :doc "free an av frame"}
    :av_frame_get_buffer {:rettype :int32
                          :check-error? true
                          :argtypes [[frame :pointer]
                                     [align :int32]]
                          :doc "Allocate the frame's data buffer"}
    :av_frame_make_writable {:rettype :int32
                             :check-error? true
                             :argtypes [[frame :pointer]]
                             :doc "Ensure frame is writable"}
    :avcodec_send_packet {:rettype :int32
                          :argtypes [[avctx :pointer] ;; AVCodecContext *avctx
                                     [avpkt :pointer?]]  ;; const AVPacket *avpkt
                          :doc "Supply raw packet data as input to a decoder.
Internally, this call will copy relevant AVCodecContext fields, which can
influence decoding per-packet, and apply them when the packet is actually
decoded. (For example AVCodecContext.skip_frame, which might direct the
decoder to drop the frame contained by the packet sent with this function.)

@warning The input buffer, avpkt->data must be AV_INPUT_BUFFER_PADDING_SIZE
         larger than the actual read bytes because some optimized bitstream
         readers read 32 or 64 bits at once and could read over the end.

@note The AVCodecContext MUST have been opened with @ref avcodec_open2()
      before packets may be fed to the decoder.

Returns: 0 on success, otherwise negative error code:
     AVERROR(EAGAIN):   input is not accepted in the current state - user
                        must read output with avcodec_receive_frame() (once
                        all output is read, the packet should be resent, and
                        the call will not fail with EAGAIN).
     AVERROR_EOF:       the decoder has been flushed, and no new packets can
                        be sent to it (also returned if more than 1 flush
                        packet is sent)
     AVERROR(EINVAL):   codec not opened, it is an encoder, or requires flush
     AVERROR(ENOMEM):   failed to add packet to internal queue, or similar
     other errors: legitimate decoding errors"}
    :avcodec_receive_frame {:rettype :int32
                            :argtypes [[avctx :pointer]
                                       [frame :pointer]]
                            :doc "Return decoded output data from a decoder.

@param avctx codec context
@param frame This will be set to a reference-counted video or audio
             frame (depending on the decoder type) allocated by the
             decoder. Note that the function will always call
             av_frame_unref(frame) before doing anything else.

@return
     0:                 success, a frame was returned
     AVERROR(EAGAIN):   output is not available in this state - user must try
                        to send new input
     AVERROR_EOF:       the decoder has been fully flushed, and there will be
                        no more output frames
     AVERROR(EINVAL):   codec not opened, or it is an encoder
     AVERROR_INPUT_CHANGED:   current decoded frame has changed parameters
                              with respect to first decoded frame. Applicable
                              when flag AV_CODEC_FLAG_DROPCHANGED is set.
     other negative values: legitimate decoding errors"}
    :avcodec_send_frame {:rettype :int32
                         :check-error? true
                         :argtypes [[ctx :pointer]
                                    [frame :pointer?]]
                         :doc "Send a frame to encode.  A nil frame flushes the buffer"}
   ;;This returns errors during normal course of operation
    :avcodec_receive_packet {:rettype :int32
                             :argtypes [[ctx :pointer]
                                        [packet :pointer]]
                             :doc "Get an encoded packet.  Packet must be unref-ed"}
    :av_packet_unref {:rettype :void
                      :argtypes [[packet :pointer]]
                      :doc "Unref a packet after from receive frame"}
    :av_frame_unref {:rettype :void
                     :argtypes [[frame :pointer]] ;;AVFrame*
                     :doc "Unreference all the buffers referenced by frame and reset the frame fields."}}
  nil
  check-error)



(defn set-library-instance!
  [lib-instance]
  (dt-ffi/library-singleton-set-instance! lib lib-instance))


(defn str-error
  [error-num]
  (let [charbuf (native-buffer/malloc 64 {:resource-type nil})]
    (try
      (let [res (av_strerror error-num charbuf 64)]
        (if (>= (long res) 0)
          (dt-ffi/c->string charbuf)
          (format "Unreconized error num: %s" error-num)))
      (finally
        (native-buffer/free charbuf)))))


(defn initialize!
  []
  (if (nil? (dt-ffi/library-singleton-library lib))
    (do
      (dt-ffi/library-singleton-set! lib "avcodec")
      :ok)
    :already-initialized))


(defn- read-codec-pixfmts
  [^long addr]
  (when-not (== addr 0)
    (let [nbuf (-> (native-buffer/wrap-address addr 4096 nil)
                   (native-buffer/set-native-datatype :int32))]
      (->> (take-while #(not= -1 %) nbuf)
           (mapv av-pixfmt/value->pixfmt)))))


(defn expand-codec
  [codec-ptr]
  (when codec-ptr
    (let [codec (dt-ffi/ptr->struct (:datatype-name @av-context/codec-def*) codec-ptr)]
      {:codec codec-ptr
       :name (dt-ffi/c->string (Pointer. (:name codec)))
       :long-name (dt-ffi/c->string (Pointer. (:long-name codec)))
       :media-type (:type codec)
       :pix-fmts (read-codec-pixfmts (:pix-fmts codec))
       :codec-id (:id codec)
       :encoder? (== 1 (long (av_codec_is_encoder codec-ptr)))
       :decoder? (== 1 (long (av_codec_is_decoder codec-ptr)))})))


(defn list-codecs
  "Iterate through codecs returning a list of codec properties"
  []
  ;;ensure struct definition
  (let [opaque (dt-ffi/make-ptr :pointer 0)]
    (->> (repeatedly #(av_codec_iterate opaque))
         (take-while identity)
         (mapv expand-codec))))


(defn find-encoder-by-name
  [name]
  (expand-codec (avcodec_find_encoder_by_name name)))


(defn find-encoder
  [codec-id]
  (expand-codec (avcodec_find_encoder codec-id)))


(defn find-decoder-by-name
  [name]
  (expand-codec (avcodec_find_decoder_by_name name)))


(defn find-decoder
  [codec-id]
  (expand-codec (avcodec_find_decoder codec-id)))


(defn alloc-context
  (^Map [codec-ptr]
   (->> (avcodec_alloc_context3 codec-ptr)
        (dt-ffi/ptr->struct (:datatype-name @av-context/context-def*))))
  (^Map []
   (alloc-context nil)))


(defn free-context
  [ctx]
  (resource/stack-resource-context
   (let [ctx-ptr (->> (dt-ffi/->pointer ctx)
                      (.address)
                      (dt-ffi/make-ptr :pointer))]
     (avcodec_free_context ctx-ptr))))


(defn alloc-packet
  ^Map []
  (->> (av_packet_alloc)
       (dt-ffi/ptr->struct (:datatype-name @av-context/packet-def*))))


(defn free-packet
  [pkt]
  (resource/stack-resource-context
   (let [pkt-ptr (->> (dt-ffi/->pointer pkt)
                      (.address)
                      (dt-ffi/make-ptr :pointer))]
     (av_packet_free pkt-ptr))))


(defn alloc-frame
  ^Map []
  (->> (av_frame_alloc)
       (dt-ffi/ptr->struct (:datatype-name @av-context/frame-def*))))


(defn free-frame
  [pkt]
  (resource/stack-resource-context
   (let [pkt-ptr (->> (dt-ffi/->pointer pkt)
                      (.address)
                      (dt-ffi/make-ptr :pointer))]
     (av_frame_free pkt-ptr))))
