(ns clj-memory-meter.core
  (:require [clojure.java.io :as io])
  (:import java.io.File
           java.lang.management.ManagementFactory
           java.net.URL))

;;;; Agent JAR unpacking

(def ^:private jamm-jar-name "jamm-0.3.2.jar")

(defn- unpack-jamm-from-resource []
  (let [dest (File/createTempFile "jamm" ".jar")]
    (io/copy (io/input-stream (io/resource jamm-jar-name)) dest)
    (.getAbsolutePath dest)))

(defonce ^:private extracted-jamm-jar (unpack-jamm-from-resource))

;;;; Agent loading

(defonce ^:private tools-jar-loaded
  (let [tools-jar (format "file:///%s/../lib/tools.jar"
                          (System/getProperty "java.home"))
        cl (.getContextClassLoader (Thread/currentThread))]
    (.addURL cl (URL. tools-jar))))

(defn- get-self-pid
  "Returns the process ID of the current JVM process."
  []
  (let [^String rt-name (.getName (ManagementFactory/getRuntimeMXBean))]
    (subs rt-name 0 (.indexOf rt-name "@"))))

#_(get-self-pid)

(defn- mk-vm [pid]
  (let [method (.getDeclaredMethod (resolve 'com.sun.tools.attach.VirtualMachine)
                                   "attach"
                                   (into-array Class [String]))]
    (.invoke method nil (object-array [pid]))))

(defn- load-jamm-agent []
  (let [vm (mk-vm (get-self-pid))]
    (.loadAgent vm extracted-jamm-jar)
    (.detach vm)
    true))

(defonce ^:private jamm-agent-loaded (load-jamm-agent))

;;;; Public API

(def ^:private memory-meter
  (.newInstance (Class/forName "org.github.jamm.MemoryMeter")))

(defn- convert-to-human-readable
  "Taken from http://programming.guide/java/formatting-byte-size-to-human-readable-format.html."
  [bytes]
  (let [unit 1024]
    (if (< bytes unit)
      (str bytes " B")
      (let [exp (int (/ (Math/log bytes) (Math/log unit)))
            pre (nth "KMGTPE" (dec exp))]
        (format "%.1f %sB" (/ bytes (Math/pow unit exp)) pre)))))

#_(convert-to-human-readable 512)
#_(convert-to-human-readable 10e8)

(defn measure
  "Shallowly measure the memory usage of the `object`. Return a human-readable
  string. Options:

  :meter  - custom org.github.jamm.MemoryMeter object
  :bytes? - if true, return a number of bytes instead of a string"
  [object & {:keys [meter bytes?]}]
  (let [m (or meter memory-meter)
        bytes (.measure m object)]
    (if bytes?
      bytes
      (convert-to-human-readable bytes))))

(defn measure-deep
  "Deeply measure the memory usage of the `object` together with all its children.
  Return a human-readable string. Options:

  :debug  - if true, print the object layout tree to stdout. Can also be set to
            a number to limit the nesting level being printed.
  :meter  - custom org.github.jamm.MemoryMeter object
  :bytes? - if true, return a number of bytes instead of a string"
  [object & {:keys [debug meter bytes?]}]
  (let [m (or meter memory-meter)
        m (cond (integer? debug) (.enableDebug m debug)
                debug (.enableDebug m)
                :else m)
        bytes (.measureDeep m object)]
    (if bytes?
      bytes
      (convert-to-human-readable bytes))))

#_(measure-deep (vec (repeat 100 "hello")) :debug true)
#_(measure-deep (object-array (repeatedly 1000 (fn [] (String. "foobarbaz")))) :bytes? true)
