(ns clj-async-profiler.flamebin
  "Code responsible for uploading profiling results to flamebin.dev."
  (:require [clj-async-profiler.post-processing :as proc]
            [clj-async-profiler.results :as results]
            [clojure.java.io :as io])
  (:import (java.net HttpURLConnection URL)))

(def flamebin-host (atom "https://flamebin.dev"))

(defn- format-upload-url [type private?]
  (format "%s/api/v1/upload-profile?format=dense-edn%s%s"
          @flamebin-host
          (if type (str "&type=" (name type)) "")
          (if private? "&private=true" "")))

(defn- gzip-dense-profile [dense-profile]
  (let [baos (java.io.ByteArrayOutputStream.)]
    (with-open [w (io/writer (java.util.zip.GZIPOutputStream. baos))]
      (binding [*print-length* nil
                *print-level* nil
                *out* w]
        (pr dense-profile)))
    (.toByteArray baos)))

(defn- upload-dense-profile [dense-profile event private?]
  (let [^bytes gzipped (gzip-dense-profile dense-profile)
        url (URL. (format-upload-url event private?))
        ^HttpURLConnection connection
        (doto ^HttpURLConnection (.openConnection url)
          (.setDoOutput true)
          (.setRequestMethod "POST")
          (.setRequestProperty "Content-Type" "application/edn")
          (.setRequestProperty "Content-Encoding" "gzip")
          (.setRequestProperty "Content-Length" (str (alength gzipped))))]
    (with-open [output-stream (.getOutputStream connection)]
      (io/copy gzipped output-stream))
    (let [status (.getResponseCode connection)
          msg (.getResponseMessage connection)
          body (slurp (.getInputStream connection))]
      (if (< status 300)
        (let [location (.getHeaderField connection "Location")
              id (.getHeaderField connection "X-Created-ID")]
          {:location location, :id id, :body body, :message msg})
        (throw (ex-info (str (str "Failed to upload profile: " msg)
                             {:status status, :body body})))))))

(defn upload-to-flamebin [run-id-or-stacks-file options]
  (let [private? (or (:private options) (:private? options))
        {:keys [stacks-file event]} (results/find-profile run-id-or-stacks-file)
        dense-profile (proc/read-raw-profile-file-to-dense-profile stacks-file)
        {:keys [location]} (upload-dense-profile dense-profile event private?)]
    (println
     (format "Uploaded %s to Flamebin: %s%s"
             stacks-file location
             (if private?
               "\nPrivate uploads don't show on the index page and require a read-password to view."
               "")))
    location))

#_(upload-to-flamebin 1 {:private? true})
