(ns clj-async-profiler.post-processing
  (:require [clojure.java.io :as io])
  (:import clojure.lang.Compiler
           java.util.HashMap))

(defn- safe-subs [^String s, ^long start, ^long end]
  (let [lng (.length s)]
    (if (and (>= start 0)
             (<= end lng)
             (<= start end))
      (.substring  s start end)
      "")))

(defn demunge-clojure-frames
  "Transform that demunges Clojure stackframes."
  [^String s]
  ;; Hand-rolling "regexps" here for better performance.
  (let [lng (.length s)
        sb (StringBuilder.)]
    (loop [last-idx 0]
      (let [nvoke (.indexOf s "nvoke" last-idx)]
        (if (> nvoke -1)
          (let [end (.indexOf s ";" nvoke)
                end (if (= end -1) lng end)
                dot (.lastIndexOf s "." nvoke)]
            (if (and (> dot last-idx))
              (let [^String method (.substring s (unchecked-inc-int dot) end)]
                (if (and (or (.equals method "invoke")
                             (.equals method "doInvoke")
                             (.equals method "invokeStatic")
                             (.equals method "invokePrim"))
                         ;; Exclude things like clojure/lang/Var.invoke
                         (let [last-slash-idx (.lastIndexOf s "/" dot)]
                           (and (> last-slash-idx -1)
                                (not (Character/isUpperCase (.charAt s (inc last-slash-idx)))))))
                  (let [start (unchecked-inc-int (.lastIndexOf s ";" dot))
                        ^String frame (.substring s start dot)
                        new-frame (.replace frame \/ \.)
                        new-frame (let [uscore (.indexOf new-frame "_")
                                        ;; Clojure demunger is slow. If there are
                                        ;; no special characters munged in the
                                        ;; frame, take a faster path.
                                        next-char-idx (inc uscore)]
                                    (if (and (<= next-char-idx (.length ^String new-frame))
                                             (Character/isUpperCase (.charAt frame next-char-idx)))
                                      (Compiler/demunge new-frame)
                                      (-> ^String new-frame
                                          (.replace \_ \-)
                                          (.replace \$ \/))))
                        ;; Check if next frame has the same name, and if so, drop it.
                        ^String possible-next-frame (.concat frame ".invokeStatic")
                        next-frame-end (min (+ end (.length possible-next-frame) 1)
                                            lng)]
                    (.append sb s last-idx start)
                    (.append sb new-frame)
                    (if (.equals possible-next-frame
                                 (safe-subs s (inc end) next-frame-end))
                      (recur next-frame-end)
                      (recur end)))
                  (do (.append sb s last-idx end)
                      (recur end))))
              (do (.append sb s last-idx end)
                  (recur end))))
          (do (.append sb s last-idx lng)
              (.toString sb)))))))

(defn post-process-stacks
  "Perform post-processing of the profiling result with the given `transform`
  function and the default processors (demunging). Write the result to
  `out-file`."
  [stacks-file out-file transform]
  (let [acc (HashMap.)]
    (with-open [f (io/reader stacks-file)]
      (loop []
        (when-let [line (.readLine ^java.io.BufferedReader f)]
          (let [sep (.lastIndexOf line " ")
                stack (subs line 0 sep)
                count (Long. (subs line (inc sep)))
                xstack (-> stack demunge-clojure-frames transform)]
            (when xstack
              (let [value (get acc xstack 0)]
                (.put acc xstack (+ value count))))
            (recur)))))

    (with-open [out (io/writer out-file)]
      (binding [*out* out]
        (run! (fn [[stack count]]
                (println stack count))
              acc)))))
