(ns ring.adapterimpl.impl
  "Adapter implementation for the Jetty 7.x webserver."
  (:import (org.eclipse.jetty.server Server Request)
         (org.eclipse.jetty.server.handler AbstractHandler)
         (org.eclipse.jetty.server.nio SelectChannelConnector)
         (org.eclipse.jetty.server.ssl SslSelectChannelConnector)
         (org.eclipse.jetty.util.thread QueuedThreadPool)
         (org.eclipse.jetty.util.ssl SslContextFactory)
         (javax.servlet.http HttpServletRequest HttpServletResponse)))

(defn idle-time-out
  "Returns idle time out for a connector. This is required since the getter
   method on jetty connectors that returns the max idle time out has been
   renamed in Jetty9x. Therefore to get unit tests to pass we need this
   wrapper function."
  [connector]
  (.getMaxIdleTime connector))

(defn- ssl-context-factory
  "Creates a new SslContextFactory instance from a map of options."
  [options]
  (let [context (SslContextFactory.)]
    (if (string? (options :keystore))
      (.setKeyStorePath context (options :keystore))
      (.setKeyStore context ^java.security.KeyStore (options :keystore)))
    (.setKeyStorePassword context (options :key-password))
    (when (options :truststore)
      (.setTrustStore context ^java.security.KeyStore (options :truststore)))
    (when (options :trust-password)
      (.setTrustStorePassword context (options :trust-password)))
    (case (options :client-auth)
      :need (.setNeedClientAuth context true)
      :want (.setWantClientAuth context true)
      nil)
    context))

(defn- ssl-connector
  "Creates a SslSelectChannelConnector instance."
  [options]
  (doto (SslSelectChannelConnector. (ssl-context-factory options))
    (.setPort (options :ssl-port 443))
    (.setHost (options :host))
    (.setMaxIdleTime (options :max-idle-time 200000))))

(defn- http-connector
  "Creates a http connection via SelectChannelConnector instance."
  [options]
  (doto (SelectChannelConnector.)
                    (.setPort (options :port 80))
                    (.setHost (options :host))
                    (.setMaxIdleTime (options :max-idle-time 200000))))

(defn create-server
  "Construct a Jetty Server instance."
  [^QueuedThreadPool tpool options]
  (let [connector (http-connector options)
        server    (doto (Server.)
                    (.addConnector connector)
                    (.setSendDateHeader true)
                    (.setThreadPool tpool))]
    (when (or (options :ssl?) (options :ssl-port))
      (.addConnector server (ssl-connector options)))
    server))
