(ns ring.adapterimpl.impl
  "Adapter implementation for the Jetty 9.x webserver."
  (:import
    (org.eclipse.jetty.server Server
                              Request
                              ServerConnector
                              HttpConfiguration
                              ConnectionFactory
                              HttpConnectionFactory
                              SslConnectionFactory
                              SecureRequestCustomizer)
         (org.eclipse.jetty.server.handler AbstractHandler)
         (org.eclipse.jetty.util.thread QueuedThreadPool)
         (org.eclipse.jetty.util.ssl SslContextFactory)
         (javax.servlet.http HttpServletRequest HttpServletResponse)))

(defn idle-time-out
  "Returns idle time out for a connector. This is required since the getter
   method on jetty connectors that returns the max idle time out has been
   renamed in Jetty9x. Therefore to get unit tests to pass we need this
   wrapper function."
  [connector]
  (.getIdleTimeout connector))

(defn- ssl-context-factory
  "Creates a new SslContextFactory instance from a map of options."
  [options]
  (let [context (SslContextFactory.)]
    (if (string? (options :keystore))
      (.setKeyStorePath context (options :keystore))
      (.setKeyStore context ^java.security.KeyStore (options :keystore)))
    (.setKeyStorePassword context (options :key-password))
    (when (options :truststore)
      (.setTrustStore context ^java.security.KeyStore (options :truststore)))
    (when (options :trust-password)
      (.setTrustStorePassword context (options :trust-password)))
    (case (options :client-auth)
      :need (.setNeedClientAuth context true)
      :want (.setWantClientAuth context true)
      nil)
    context))

(defn- ssl-connector
  "Creates a ssl connector instance."
  [server options]
  ;jetty 9 sample code indicates that one should create both http and ssl
  ;connection factories.
  ;See http://git.eclipse.org/c/jetty/org.eclipse.jetty.project.git/tree/examples/
  ;embedded/src/main/java/org/eclipse/jetty/embedded/ManyConnectors.java
  ;I do not think this is necessary.
  (let [http-config (doto (HttpConfiguration.)
                      (.addCustomizer (SecureRequestCustomizer.)))
        connection-factory (HttpConnectionFactory. http-config)
        ctx-factory (ssl-context-factory options)
        ssl-connection-factory (SslConnectionFactory. ctx-factory "http/1.1")
        sc (ServerConnector.
             server (into-array ConnectionFactory
                                [ssl-connection-factory connection-factory]))]
    (doto sc
      (.setPort (options :ssl-port 443))
      (.setHost (options :host))
      (.setIdleTimeout (options :max-idle-time 200000)))))

(defn- http-connector
  "Creates a HTTP channel connection factory instance."
  [server options]
  (let [connection-factory (HttpConnectionFactory. (HttpConfiguration.))
        sc (ServerConnector.
            server (into-array ConnectionFactory [connection-factory]))]
    (doto sc
      (.setPort (options :port 80))
      (.setHost (options :host))
      (.setIdleTimeout (options :max-idle-time 200000)))))

 (defn create-server
   "Construct a Jetty Server instance."
  [thread-pool options]
  (let [server (Server. thread-pool)]
    (.addConnector server (http-connector server options))
     (when (or (options :ssl?) (options :ssl-port))
      (.addConnector server (ssl-connector server options)))
     server))
