(ns ring.adapter.jetty
  "Adapter for the Jetty webserver."
  (:import (org.eclipse.jetty.server Server Request)
           (org.eclipse.jetty.server.handler AbstractHandler)
           (org.eclipse.jetty.util.thread QueuedThreadPool)
           (javax.servlet.http HttpServletRequest HttpServletResponse))
  (:require [ring.util.servlet :as servlet]
            [ring.adapterimpl.impl :as impl]))

(defn- proxy-handler
  "Returns an Jetty Handler implementation for the given Ring handler."
  [handler]
  (proxy [AbstractHandler] []
    (handle [_ ^Request base-request request response]
      (let [request-map  (servlet/build-request-map request)
            response-map (handler request-map)]
        (when response-map
          (servlet/update-servlet-response response response-map)
          (.setHandled base-request true))))))

(defn idle-time-out
  "Returns idle time out for a connector. This is required since the getter
   method on jetty connectors that returns the max idle time out has been
   renamed in Jetty9x. Therefore to get unit tests to pass we need this
   wrapper function."
  [connector]
  (impl/idle-time-out connector))

(defn ^Server run-jetty
  "Start a Jetty webserver to serve the given handler according to the
  supplied options:

  :configurator - a function called with the Jetty Server instance
  :port         - the port to listen on (defaults to 80)
  :host         - the hostname to listen on
  :join?        - blocks the thread until server ends (defaults to true)
  :daemon?      - use daemon threads (defaults to false)
  :ssl?         - allow connections over HTTPS
  :ssl-port     - the SSL port to listen on (defaults to 443, implies :ssl?)
  :keystore     - the keystore to use for SSL connections
  :key-password - the password to the keystore
  :truststore   - a truststore to use for SSL connections
  :trust-password - the password to the truststore
  :max-threads  - the maximum number of threads to use (default 50)
  :min-threads  - the minimum number of threads to use (default 8)
  :max-idle-time  - the maximum idle time in milliseconds for a connection (default 200000)
  :client-auth  - SSL client certificate authenticate, may be set to :need,
                  :want or :none (defaults to :none)"
  [handler options]
  (let [^QueuedThreadPool p (QueuedThreadPool. ^Integer (options :max-threads 50))
        ^Server s (impl/create-server p (dissoc options :configurator))]
    (.setMinThreads p (options :min-threads 8))
    (when (:daemon? options false)
      (.setDaemon p true))
    (doto s
      (.setHandler (proxy-handler handler)))
    (when-let [configurator (:configurator options)]
      (configurator s))
    (.start s)
    (when (:join? options true)
      (.join s))
    s))
