(ns cb.cljbeat.ottla.producer
  "An opinionated wrapper around KafkaProducer."
  (:require [clojure.walk :as walk]
            [cb.cljbeat.ottla.cli :as cli]
            [clojure.tools.logging :as log]
            [clojurewerkz.propertied.properties :as p])
  (:import [java.util Properties]
           [org.apache.kafka.clients.producer Producer]
           [org.apache.kafka.clients.producer KafkaProducer]
           [org.apache.kafka.clients.producer ProducerConfig]
           [org.apache.kafka.clients.producer ProducerRecord]))


(def PRODUCER_CONFIGURABLES
  "Producer options defined and mapped into PRODUCER_DEFAULT_PROPS (used by
  producer function) and PRODUCER_CLI_OPTION_SPECS (which can be required for
  pairing with user-defined cli-options when).

  Many options were skipped, I've only added ones that seem useful at this time.

  For more detailed descriptions, see:
  https://kafka.apache.org/documentation/#producerconfigs"

   ;; duplicate of that in consumer/CONSUMER_OPTIONS
  [{:prop "bootstrap.servers"
    :short-opt "z"
    :long-opt "ottla.producer.bootstrap.servers"
    :missing "ottla.producer.bootstrap.servers must be specified"
    :hint "HOST:9092,HOST:9092"
    :desc "Broker(s) to connect to."}

   {:prop "key.serializer"
    :long-opt "ottla.producer.key.serializer"
    :desc "Serializer class for key that implements the Serializer interface."
    :default "org.apache.kafka.common.serialization.StringSerializer"}

   {:prop "value.serializer"
    :long-opt "ottla.producer.value.serializer"
    :desc "Serializer class for value that implements the Serializer interface."
    :default "org.apache.kafka.common.serialization.ByteArraySerializer"}

   {:prop "acks"
    :long-opt "ottla.producer.acks"
    :desc "Num acks leader required to receive before considering req complete."
    :default "1"}

   {:prop "retries"
    :long-opt "ottla.producer.retries"
    :desc "Retry failed sends. This can cause dupes but it can also allow recovery from exceptions when partition leadership changes."
    :default "10"}

   {:prop "buffer.memory"
    :long-opt "ottla.producer.buffer.memory"
    :desc "Bytes memory the producer can use to buffer records before sending."
    :default "33554432"}

   {:prop "compression.type"
    :long-opt "ottla.producer.compression.type"
    :desc "Compression for data generated by producer (none, gzip, snappy, lz4)"
    :default "none"}

   {:prop "batch.size"
    :long-opt "ottla.producer.batch.size"
    :desc "Max bytes of batching of records on a per-partition basis."
    :default "16384"}

   ;; would probably be useful to set this to the aurora job id
   {:prop "client.id"
    :long-opt "ottla.producer.client.id"
    :desc "An id string to pass to the server when making requests."
    :default ""}

   ;; we might want to play with this in the future...
   {:prop "partitioner.class"
    :long-opt "ottla.producer.partitioner.class"
    :desc "Partitioner class that implements the Partitioner interface."
    :default "org.apache.kafka.clients.producer.internals.DefaultPartitioner"}])

(def PRODUCER_VALID_PROPS (into [] (map #(get % :long-opt) PRODUCER_CONFIGURABLES)))

(def PRODUCER_DEFAULT_PROPS
  "The map of default properties created from PRODUCER_CONFIGURABLES"
  (into {} (map (juxt :prop :default) PRODUCER_CONFIGURABLES)))


(def PRODUCER_CLI_OPTIONS
  "A list of cli-option-specs to be used in conjunction with cli/parse-opts and
  with producer/extract-props-from-options."
  (map cli/configurable-map->cli-opt-vect PRODUCER_CONFIGURABLES))


(def extract-props-from-options
  "Converts a map of options created by parsing PRODUCER_CLI_OPTIONS into a map
  of properties for the producer"
  (partial cli/extract-props-from-options PRODUCER_CONFIGURABLES))

(defn safe-props
  [m]
  (persistent!
    (reduce-kv (fn [acc k v]
                 (if (string? v)
                   (conj! acc [k v])
                   acc))
               (transient {})
               m)))


(defn producer
  [props]
  (let [;; apply defaults, java-fy props, allow keyword keys
        props (walk/stringify-keys props)
        props (into PRODUCER_DEFAULT_PROPS props)]
      (KafkaProducer. (p/load-from props))))


(defn send!
  "Sends one message with value to the KafkaProducer prdcr on the String topic
  with the partition determined by key.

  The type and then serialization of key and value are dictated by the producer
  properties key.serializer and value.serializer which are defaulted to
  StringSerializer and ByteArraySerializer respectively.

  DOES NOT GUARANTEE A WRITE.

  Returns a future which, after write happens, contains a RecordMetadata. To
  ensure write to topic, flush! must be called."
  ([^Producer prdcr ^String topic key value]
    (.send prdcr (ProducerRecord. topic key value)))
  ([^Producer prdcr ^String topic ^Integer partition key value]
    (.send prdcr (ProducerRecord. topic partition key value))))


(defn flush!
  "Makes all buffered records immediately available to send (even if linger.ms
  is greater than 0) and blocks until complete."
  [^Producer prdcr]
  (.flush prdcr))


(defn send-and-flush-batch!
  "Combines send! and flush! to send a batch of messages defined by a seq of
  maps that each have a :topic, :key, :value and optionally a :partition to put the message on.
  "
  [^Producer prdcr messages]
  (let [;; send! each message, this gives us a not-lazy seq of futures
        sent (->> (for [{t :topic p :partition k :key v :value} messages]
                    (if (some? p)
                      (send! prdcr t p k v)
                      (send! prdcr t k v)))
                  (doall))]

    ;; flush the messages out of the producer (and any other messages that may
    ;; be buffered)
    (flush! prdcr)

    ;; block on completion of futures to ensures the writes happend and expose the
    ;; RecordMetadatas to the user
    (mapv deref sent)))
