(ns currency-code-validator.core
  (:require [clojure.string :as str]
            [clojure.java.io :as io]
            [clojure.pprint :as pp]
            [clojure.xml :as xml])
  (:gen-class))


(def data (atom #{}))

(defn load-data []
  ;; original file
  ;; https://www.currency-iso.org/dam/downloads/lists/list_one.xml
  ;;
  ;; Saved to resources 04-08-2019
  (let [m (-> "data-file.xml"
              io/resource
              io/input-stream
              xml/parse)]
    (let [m2 (filter #(= 5 (count (:content %))) (-> m
                                                     :content
                                                     first
                                                     :content))]
      (->> m2
           (mapv (fn [m] (let [content (:content m)]
                          {:code (first (:content (nth content 2)))
                           :name (first (:content (nth content 0)))})))))))

(defn ensure-loaded []
    (if (empty? @data) (reset! data (load-data))))


(defn currency-codes []
  (ensure-loaded)
  (sort (into #{} (map :code @data))))

(defn valid [code]
  (ensure-loaded)
  (let [s (into #{} (map :code @data))]
    (s (str/upper-case code))))


(defn all []
  (ensure-loaded)
  (sort-by :code @data))

(defn all-name []
  (ensure-loaded)
  (sort-by :name @data))

(defn debug-print [s]
  (println (format "Is currency code %s valid? %b" s (valid s))))


(defn -main [& args]
  (println "--------------------------------------------------")
  (debug-print "USD")
  (debug-print "XXY")
  (println "--------------------------------------------------")
  (doseq [m (all)]
    (println (format "%s - %s" (:code m) (:name m)))))
