(ns country-code-validator.core
  (require [clojure.string :as str]
           [clojure.java.io :as io])
  (:gen-class))


(def country-codes (atom #{}))

(defn load-data []
  ;; original file
  ;; https://web.archive.org/web/20161122071627/http://www.iso.org/iso/home/standards/country_codes/country_names_and_code_elements_txt-temp.htm
  ;;
  ;; Saved to resources 04-08-2019
  (let [data (-> "data-file.txt"
                 io/resource
                 io/input-stream
                 slurp (str/split #"\n")
                 rest)]
    (->> data
         (map (fn [s] (let [[n c] (str/split s #";")]
                        {:code c
                         :name n})))
         (sort-by :code))))

(defn ensure-loaded []
  (if (empty? @country-codes) (reset! country-codes (load-data))))

(defn valid [s]
  (ensure-loaded)
  (let [codes (into #{} (map :code @country-codes))]
    (codes (str/upper-case s))))

(defn all []
  (ensure-loaded)
  @country-codes)

(defn debug-print [s]
  (println (format "Is country code %s valid? %b" s (valid s))))

(defn -main [& args]
  (println "--------------------------------------------------")
  (debug-print "US")
  (debug-print "XX")
  (println "--------------------------------------------------")
  (doseq [m (all)]
    (println (format "%s - %s" (:code m) (:name m)))))

