(ns bizlogic.tools.util
  (:require [clojure.string :as str]
            [clojure.string :as string]
            [io.pedestal.interceptor :as interceptor
             :refer [definterceptorfn defon-request]]
            [io.pedestal.impl.interceptor :as int-impl]
            [io.pedestal.http.route :as proute]
            [bizlogic.tools.log :as log]
            [ring.util.response :as ring-response]
            [ring.middleware.file :as rmfile]
            [ring.middleware.edn :as edn]
            [clojure.java.io :as io]
            [datomic.api :as d :refer [db]])
  (:import (java.io File FileOutputStream BufferedOutputStream
             ByteArrayOutputStream)
           (java.util.zip ZipInputStream)
           (java.util.jar JarFile Manifest JarEntry JarOutputStream)
           (java.util.regex Pattern)))

(defn insert-at [n x coll]
  (vec (concat (take n coll) [x] (drop (inc n) coll))))

(defn- ensure-ends-with-sep [p]
  (if (.endsWith p File/separator) p (str p File/separator)))

(defn trim-left-right [coll]
  (map (fn [x]
         (-> (#(str/replace-first (str/reverse x) "-" ""))
             (#(str/replace-first (str/reverse %) "-" ""))))
       coll))

(def email-re #"\b[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,4}\b")

#_
(defn name [x]
  "Returns empty string when the arg is nil to avoid throwing
   NullPointerException."
  (if x (clojure.core/name x) ""))

(defn- ensure-dir
  "Ensures that a directory exists at the given path, throwing if one does not."
  [^String dir-path]
  (let [dir (java.io.File. dir-path)]
    (if-not (.exists dir)
      (throw (Exception. (format "Directory does not exist: %s" dir-path))))))

(defn- edn-request?
  [req]
  (if-let [^String type (:content-type req)]
    (not (empty? (re-find #"^application/(vnd.+)?edn" type)))))

(defon-request edn-params
  [request]
  (if-let [body (and (edn-request? request) (:body request))]
    (let [edn-params (binding [*read-eval* false] (edn/-read-edn body))]
      (assoc request
        :edn-params edn-params
        :params (merge (:params request) edn-params)))
    request))

(interceptor/defon-request log-request
  "Log the request's method and uri."
  [request]
  (log/info
    :msg "logging request without servlets"
    :request request)
  request)

(interceptor/defon-request log-request-headers
  "Log the request's method and uri."
  [request]
  (log/info
    :msg (format "%s %s"
                         (string/upper-case (name (:request-method request)))
                         (:uri request))
    :headers (:headers request))
  request)

(interceptor/definterceptorfn log-request-key [keys]
  (interceptor/on-request
    (fn [request]
      (let [key-path (symbol (str/join "/" keys))]
        (log/info :msg (str "logging request at " key-path)
          key-path (get-in request keys))))))

(interceptor/defon-response log-response
  "Log the request's method and uri."
  [response]
  (log/info :msg "Final response map"
    :response response)
  response)

(interceptor/defaround log-session
  ([ctx]
    (log/info :msg "session before"
      :session (get-in ctx [:request :session]))
    ctx)
  ([ctx]
    (log/info :msg "session after"
      :session (get-in ctx [:response :session]))
    ctx))

(interceptor/defbefore log-context-before
  "log the context"
  [context]
  (log/info :msg "Logging the context"
    :context context)
  context)

(interceptor/defafter log-context-after
  "log the context"
  [context]
  (log/info :msg "Logging the context"
    :context context)
  context)

(interceptor/definterceptorfn  log-context-key [keys]
  (interceptor/around
    (fn [ctx]
      (let [key-path (symbol (str/join "/" keys))]
        (log/info :msg (str "logging ctx at " key-path)
          key-path (get-in ctx keys))))
    (fn [ctx]
      (let [key-path (symbol (str/join "/" keys))]
        (log/info :msg (str "logging ctx at " key-path)
          key-path (get-in ctx keys))))))

(interceptor/definterceptorfn log-key-middleware [keys]
  (interceptor/middleware
    ::log-key-middleware
    (fn [request]
      (let [key-path (symbol (str/join "/" keys))]
        (log/info :msg (str "logging request at " key-path)
          key-path (get-in request keys))
        request))
    (fn [response]
      (let [key-path (symbol (str/join "/" keys))]
        (log/info :msg (str "logging response at " key-path)
          key-path (get-in response keys))
        response))))

(defn- response?
  "A valid response is any map that includes an integer :status
  value."
  [resp]
  (and (map? resp)
       (integer? (:status resp))))

(interceptor/defafter not-found
  "An interceptor that returns a 404 when routing failed to resolve a route."
  [context]
  (if-not (response? (:response context))
    (do (log/info :in :not-found :response (:response context))
        (assoc context
          :response (ring-response/not-found (pr-str "Not Found!!"))))
    context))

(interceptor/defbefore print-request
  [{:keys [request] :as ctx}]
  (let [intcpt (:name (peek (pop (::int-impl/stack ctx))))]
    (println (format "Printing request after Interceptor : %s " intcpt))
    (println request)
    ctx))


(definterceptorfn file
  "Interceptor for file ring middleware."
  [root-path & [opts]]
  (interceptor/handler ::file
    #(if-let [response (rmfile/file-request % root-path opts)]
       response
       %)))

;; possibly make maniffest from project.clj or some other config
(defn jar [dir target & manifest]
  (let [zipinput (ZipInputStream. dir)
        file-output (io/file target)
        jar-output (-> file-output
                     (FileOutputStream.)
                     (BufferedOutputStream.)
                     (JarOutputStream. manifest))
        zip-entries (cond
                      )]))

(def tomcat-webapps "/opt/tomcat/tomcat7/base/webapps")

(defn deploy-tomcat [jarname])

(definterceptorfn after-router [routes-fn]
  (interceptor/after
    ::after-router
    (fn [context]
      (let [routes (routes-fn)]
        (@#'proute/route-context-to-matcher-routes context
          (mapv #(assoc % :matcher (@#'proute/matcher %)) routes)
          routes)))))

;; datomic
(defn build-schema [ident type card & [doc unique index]]
  (merge {:db/id #db/id[:db.part/db]
          :db/ident ident
          :db/valueType (keyword (str "db.type/" (name type)))
          :db/cardinality (keyword (str "db.cardinality/" (name card)))}
    (if doc {:db/doc doc})
    (or (and unique {:db/unique (keyword (str "db.unique/" unique))})
      (or (and index {:db/index true}) nil))))
