(ns bizlogic.tools.session
  (:require [datomic.api :as d :refer [q db]]
            [ring.middleware.session.store :as store]
            [clj-uuid])
  (:import java.util.UUID))

(defonce session-uri "datomic:mem://session")

(d/create-database session-uri)

(def session-conn (d/connect session-uri))

(def session-schema  [{:db/id (d/tempid :db.part/db)
                       :db/ident :session/key
                       :db/valueType :db.type/uuid
                       :db/unique :db.unique/value
                       :db/cardinality :db.cardinality/one
                       :db/doc "Session Key"
                       :db.install/_attribute :db.part/db}
                      {:db/id (d/tempid :db.part/db)
                       :db/ident :csrf/anti-forgery-token
                       :db/valueType :db.type/string
                       :db/unique :db.unique/value
                       :db/cardinality :db.cardinality/one
                       :db/doc "CSRF Token"
                       :db.install/_attribute :db.part/db}])

@(d/transact session-conn session-schema)

(defn new-session-id []
  (clj-uuid/v4))

(defn session-store* [conn]
  (reify
    store/SessionStore
    (read-session [store key]
      (println "session/key is: " key)
      (let [db (d/db conn)]
        (if-let [e (and (clj-uuid/uuid? (str key))
                     (d/entity db
                       (ffirst (d/q '[:find ?e
                                      :in $ ?uuid
                                      :where
                                      [?e :session/key ?uuid]]
                                 db (UUID/fromString (str key))))))]
          (d/touch e)
          {})))
    (write-session [store key session-data]
      (try
        (if-let [key (or (and (clj-uuid/uuid? (str key))
                           (UUID/fromString (str key))))]
          (do @(d/transact conn
                 [(merge {:db/id [:session/key key]}
                    session-data)])
              key)
          (let [key (d/squuid)]
            @(d/transact conn [(merge {:db/id (d/tempid :db.part/user)
                                       :session/key key}
                                 session-data)])
            key))
        ;; TODO: complete with message
        (catch Exception e)))
    (delete-session [store key]
      (d/transact conn [[:db/retract :session/key key]]))))

(defonce session-store (session-store* session-conn))
