(ns bizlogic.tools.csrf
  "CSRF protection interceptor support, compatible with ring-anti-forgery"
  (:require [crypto.random :as random]
            [crypto.equality :as crypto]
            [io.pedestal.interceptor :as interceptor :refer [definterceptorfn around]]))

;; This is a port of the ring-anti-forgery CSRF protection, with the following
;; differences:
;;  * Optionally include (and handle) a double-submit cookie
;;  * A function to get the current token (to embed in a head/meta tag, html form)
;;    given the context
;;  * CSRF token is in the `request` not in a dynamic var
;;  * Structured for Pedestal interceptor
;;  * No third-party dependency on an HTML templating library


;; this is both the marker and the function (to use with the context)
(def anti-forgery-key ::anti-forgery-token)

(defn stored-token [request]
  (get-in request [:session anti-forgery-key]))

(defn new-token []
  (random/base64 60))

(defn- request-token [request]
  (get-in request [:cookies "__anti-forgery-token"]))

(defn- assoc-forgery-token [response request token]
  (let [stored-token (stored-token request)]
    (if (= stored-token token)
      response
      (assoc-in response [:session anti-forgery-key] token))))

;; This must be run after the session token setting
(defn- double-submit-cookie [request response]
  (let [token (get-in request [:session anti-forgery-key])]
    ;; The token should also be in a cookie for JS (proper double submit)
    (assoc-in response [:cookies "__anti-forgery-token"] token)))

(defn- form-params [request]
  (merge (:form-params request)
         (:multipart-params request)))

(defn- default-request-token [request]
  (or (-> request form-params (get "__anti-forgery-token"))
      (-> request :headers (get "x-csrf-token"))
      (-> request :headers (get "x-xsrf-token"))))

(defn- valid-request? [request read-token]
  (let [request-token   (read-token request)
        stored-token (stored-token request)]
    (if stored-token
      (and request-token
        (crypto/eq? request-token stored-token))
      true)))

(defn- get-request? [{method :request-method :as request}]
  (or (= :head method)
      (= :get method)))

(defn access-denied-response [body]
  {:status 403
   :headers {"Content-Type" "text/html"}
   :body body})

(def denied-msg "<h1>Invalid anti-forgery token</h1>")

(def default-error-response (access-denied-response denied-msg))

(definterceptorfn anti-forgery
  "Interceptor that prevents CSRF attacks. Any POST/PUT/PATCH/DELETE request to
  the handler returned by this function must contain a valid anti-forgery
  token, or else an access-denied response is returned.

  The anti-forgery token can be placed into a HTML page via the
  ::anti-forgery-token within the request, which is bound to a random key
  unique to the current session. By default, the token is expected to be in a
  form field named '__anti-forgery-token', or in the 'X-CSRF-Token' or
  'X-XSRF-Token' headers.

  This behavior can be customized by supplying a map of options:
    :read-token
      a function that takes a request and returns an anti-forgery token, or nil
      if the token does not exist.
    :cookie-token
      a truthy value, if you want a CSRF double-submit cookie set
    :error-response
      the response to return if the anti-forgery token is incorrect or missing.
    :error-handler
      a handler function (passed the context) to call if the anti-forgery
      token is incorrect or missing (intended to return a valid response).

  Only one of :error-response, :error-handler may be specified."
  ([] (anti-forgery {}))
  ([options]
   {:pre [(not (and (:error-response options)
                    (:error-handler options)))]}
   (let [double-submit? (:double-submit? options)
         error-response (:error-response options default-error-response)
         error-handler (:error-handler options (fn [context]
                                                 (assoc-in context [:response] error-response)))]
     (around ::anti-forgery
       (fn [{request :request :as context}]
         (let [stored-token (stored-token request)]
           (if (and (get-request? request))
             context
             (if (valid-request? request default-request-token)
               (assoc-in context [:session anti-forgery-key]
                 (or stored-token (new-token)))
               ;; TODO: log the error!
               (error-handler context)))))
       (fn [{response :response req :request :as context}]
         (let [stored-token (stored-token req)]
           (assoc context
             :response (cond-> (assoc-forgery-token response req stored-token)
                         double-submit? (#(double-submit-cookie req %))))))))))
