(ns hom.core
  (:require
    [clojure.string :as str]
    [om.next :as om]
    [om.dom :as dom]))

(defn- parse-element
  "Returns a map containing HTML attributes from a Hiccup element."
  [el]
  (let [parts (str/split (name el) #"\.")
        [tag id] (str/split (first parts) #"#")
        class-name (str/join " " (rest parts))]
    (merge {:tag tag}
           (when id {:id id})
           (when-not (str/blank? class-name) {:className class-name}))))

#?(:cljs
   (defn component
     "Returns a React component from a one dimensional vector of Hiccup data."
     [obj]
     (if-not (vector? obj)
       obj
       (let [[el & args] obj
             opts? (map? (first args))
             children (if opts? (rest args) args)
             {:keys [tag] :as parts} (parse-element el)
             om-dom-fn (aget om.dom tag)
             attrs (merge {}
                          (when opts? (first args))
                          (select-keys parts [:id :className]))]
         (apply om-dom-fn
                (when-not (empty? attrs)
                  (cljs.core/clj->js attrs))
                (reduce (fn [memo child]
                          (cond
                            (vector? child) (conj memo (component child))
                            (seq? child) (concat memo (map component child))
                            :else (conj memo child)))
                        []
                        children))))))

(defmacro defui
  [& body]
  `(om.next/defui ~@(map (fn [form]
                           (if (and (seq? form)
                                    (= 'render (first form)))
                             (list 'render '[this]
                                   (conj (drop 2 form) 'hom.core/component))
                             form))
                         body)))
