(ns hom.core
  (:require 
    [clojure.string :as str]
    [om.next :as om]
    [om.dom :as dom]))

(defmulti expand-form {:private true} #(first %))

(defn- parse-element [el]
  (let [parts (str/split (name el) #"\.")
        [tag id] (str/split (first parts) #"#")
        class-name (str/join " " (rest parts))]
    (merge {:tag tag}
           (when id {:id id})
           (when-not (str/blank? class-name) {:className class-name}))))

(defn- rewrite [f body]
  `~(cond
      (vector? body) (apply f body)
      (seq? body) (expand-form body)
      :else body))

(defn- expand-element
  [el & args]
  (let [opts? (map? (first args))
        body (if opts? (rest args) args)
        nxt (first body)
        {:keys [tag] :as parts} (parse-element el)
        attrs (merge {}
                     (when opts? (first args))
                     (select-keys parts [:id :className]))]
    `(~(symbol "om.dom" tag)
               ~(when-not (empty? attrs)
                  `(cljs.core/clj->js ~attrs))
               ~@(map (fn [form]
                        (cond
                          (vector? form) (apply expand-element form)
                          (seq? form) (expand-form form)
                          :else form))
                      body))))

(defmethod expand-form 'let [[_ forms body]]
  `(let ~forms ~@(if (seq? body)
                  (map #(rewrite expand-element %) body)
                  (list (rewrite expand-element body)))))

(defmethod expand-form 'when [[_ condition body]]
  `(when ~condition ~(expand-element body)))

(defmethod expand-form :default [frm]
  `~frm)

(defmacro component [conf]
  (rewrite expand-element conf))

(defmacro defui [& body]
  `(om.next/defui ~@(map (fn [form]
                           (if (and (seq? form)
                                    (= 'render (first form)))
                             (list 'render '[this]
                                   (conj (drop 2 form) 'hom.core/component))
                             form))
                         body)))
